///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file ImportExportManager.h 
 * \brief Contains the definition of the Core::ImportExportManager class. 
 */

#ifndef __OVITO_IMPORT_EXPORT_MANAGER_H
#define __OVITO_IMPORT_EXPORT_MANAGER_H

#include <core/Core.h>
#include "ImporterExporter.h"

namespace Core {

/// \def IMPORTEXPORT_MANAGER
/// \brief The predefined instance of the Core::ImportExportManager class.
/// 
/// Always use this macro to access the Core::ImportExportManager class instance.
#define IMPORTEXPORT_MANAGER		(*ImportExportManager::getSingletonInstance())

/**
 * \brief This descriptor contains information about an installed ImporterExporter service.
 * 
 * The list of installed import services can be retrieved using the ImportExportManager::importers() method.
 * 
 * The list of installed export services can be retrieved using the ImportExportManager::exporters() method.
 * 
 * \author Alexander Stukowski
 * \sa ImporterExporter, ImportExportManager
 */
class CORE_DLLEXPORT ImportExportDescriptor 
{
public:
	
	/// \brief Returns the file filter that specifies the files that can be 
	///        imported/exported by this service.
	/// \return The wildcard string that specifies the file types that can be handled by this import 
	///         or export plugin class. 
	const QString& fileFilter() const { return _fileFilter; }
	
	/// \brief Returns the filter description that is displayed in the dropdown of the file dialog.
	/// \return A string that describes the supported file type.
	const QString& fileFilterDescription() const { return _fileFilterDescription; }
	
	/// \brief Creates an instance of the import/export plugin class.
	/// \return A new instance of the plugin class that can be used to import or export
	///         from and to files.
	ImporterExporter::SmartPtr createService() const {
		return static_object_cast<ImporterExporter>(pluginClass()->createInstance());
	}
	
	/// \brief Returns the plugin class descriptor for the import/export service.
	/// \return The descriptor of the ImporterExporter derived class.
	PluginClassDescriptor* pluginClass() const { return _pluginClass; }
	
private:
	
	QString _fileFilter;
	QString _fileFilterDescription;
	PluginClassDescriptor* _pluginClass;
	
	friend class ImportExportManager;
};

/**
 * \brief Manages the installed import and export services.
 * 
 * This is a singleton class with only one predefined instance of this class. 
 * You can access the instance of this class using the IMPORTEXPORT_MANAGER macro.
 * 
 * \author Alexander Stukowski
 * \sa ImporterExporter, ImportExportDescriptor
 */
class CORE_DLLEXPORT ImportExportManager : public QObject
{
	Q_OBJECT
public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the ImportExportManager singleton class.
	/// \note You should use the IMPORTEXPORT_MANAGER macro to access the ImportExportManager instance instead
	///       of this method.
	inline static ImportExportManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance, "ImportExportManager::getSingletonInstance", "ImportExportManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// Return the descriptors for all installed import plugins.
	const QVector<ImportExportDescriptor>& importers() const { return _importers; }

	/// Return the descriptors for all installed export plugins.
	const QVector<ImportExportDescriptor>& exporters() const { return _exporters; }

	/// \brief Tries to detect the format of the given file.
	/// \return The import plugin that can handle this file. If the file format could not be recognized then NULL is returned.
	ImporterExporter::SmartPtr autodetectFileFormat(const QString& file);

private:

	/// Contains the description of all installed import plugins.
	QVector<ImportExportDescriptor> _importers;

	/// Contains the description of all installed export plugins.
	QVector<ImportExportDescriptor> _exporters;

private:
    
	/// Private constructor.
	/// This is a singleton class; no public instances are allowed.
	ImportExportManager();

	/// Initializes the ImportExportManager class.
	/// This is called at program startup.
	static void initialize() { 
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new ImportExportManager();
	}
	
	/// Releases the ImportExportManager.
	/// This is called at program shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}

	/// The singleton instance of this class.
	static ImportExportManager* _singletonInstance;

	friend class ApplicationManager;
};

};

#endif // __OVITO_IMPORT_EXPORT_MANAGER_H
