type token =
  | TokEof of (Lm_location.t)
  | TokEol of (Lm_location.t)
  | TokWhite of (string * Lm_location.t)
  | TokLeftParen of (string * Lm_location.t)
  | TokRightParen of (string * Lm_location.t)
  | TokArrow of (string * Lm_location.t)
  | TokComma of (string * Lm_location.t)
  | TokColon of (string * Lm_location.t)
  | TokDoubleColon of (string * Lm_location.t)
  | TokNamedColon of (string * Lm_location.t)
  | TokDollar of (string * Omake_ast.apply_strategy * Lm_location.t)
  | TokEq of (string * Lm_location.t)
  | TokArray of (string * Lm_location.t)
  | TokDot of (string * Lm_location.t)
  | TokId of (string * Lm_location.t)
  | TokKey of (string * Lm_location.t)
  | TokKeyword of (string * Lm_location.t)
  | TokCatch of (string * Lm_location.t)
  | TokClass of (string * Lm_location.t)
  | TokOp of (string * Lm_location.t)
  | TokInt of (string * Lm_location.t)
  | TokFloat of (string * Lm_location.t)
  | TokString of (string * Lm_location.t)
  | TokBeginQuote of (string * Lm_location.t)
  | TokEndQuote of (string * Lm_location.t)
  | TokBeginQuoteString of (string * Lm_location.t)
  | TokEndQuoteString of (string * Lm_location.t)
  | TokStringQuote of (string * Lm_location.t)
  | TokVar of (Omake_ast.apply_strategy * string * Lm_location.t)
  | TokVarQuote of (Omake_ast.apply_strategy * string * Lm_location.t)

open Parsing;;
let _ = parse_error;;
# 3 "omake_ast_parse.mly"
include Omake_pos.Make (struct let name = "Omake_parse" end)


(*
 * Define flags.
 *)
let define_flag (s, loc) =
   match s with
    |  "=" -> Omake_ast.DefineNormal
    | "+=" -> DefineAppend
    | _ ->
       raise (Omake_value_type.OmakeException (loc_exp_pos loc, StringStringError ("undefined assignment operator", s)))

(*
 * Convert arguments to parameters.
 *)
let key_of_id s =
   Lm_symbol.add (String.sub s 1 (String.length s - 1))

let parse_id_param s loc =
   match s.[0] with
      '?' -> Omake_ast.OptionalParam (key_of_id s, NullExp loc, loc)
    | '~' -> RequiredParam (key_of_id s, loc)
    | _   -> NormalParam (Lm_symbol.add s, loc)

let param_of_arg arg =
   match arg with
   | Omake_ast.IdArg (s, _, loc) ->
        parse_id_param s loc
   | NormalArg (KeyArg (v, e)) ->
        OptionalParam (v, e, Omake_ast_util.loc_of_exp e)
   | NormalArg (ExpArg e) ->
        raise (Omake_value_type.OmakeException (loc_exp_pos (Omake_ast_util.loc_of_exp e), StringAstError ("illegal function parameter", e)))
   | NormalArg (ArrowArg (_, e)) ->
        raise (Omake_value_type.OmakeException (loc_exp_pos (Omake_ast_util.loc_of_exp e), StringAstError ("illegal function argument", e)))

let get_fun_params args =
   List.map param_of_arg args

(*
 * Remove the IdArg.
 *)
let arg_of_parse_arg = function
   | Omake_ast.IdArg (s, w, loc1) ->
      let id = Omake_ast.StringIdExp (s, loc1) in
      let e =
         match w with
             Some (w, loc2) ->
                Omake_ast.SequenceExp ([id; StringWhiteExp (w, loc2)], loc1)
           | None ->
                id
      in
          Omake_ast.ExpArg e
 | NormalArg arg ->
      arg

let args_of_parse_args = List.map arg_of_parse_arg

(*
 * Utilities.
 *)
let rec simplify e =
   match e with
   | Omake_ast.SequenceExp ([e], _) ->
         simplify e
   | _ -> e

let sequence_exp l loc =
   match l with
   | [e] ->
         e
   | _ ->
         Omake_ast.SequenceExp (l, loc)


(*
 * Intern the method name.
 *)
let method_id_intern idl =
   List.map Lm_symbol.add idl

(*
 * Get a string from a method name.
 *)
let method_id_string idl =
   let buf = Buffer.create 32 in
   let rec collect idl =
      match idl with
         [id] ->
            Buffer.add_string buf id
       | id :: idl ->
            Buffer.add_string buf id;
            Buffer.add_char buf '.';
            collect idl
       | [] ->
            ()
   in
      collect idl;
      Buffer.contents buf

let rec method_id_rev_sequence loc items idl =
   match idl with
      [id] ->
         (Omake_ast.StringIdExp (id, loc)) :: items
    | id :: idl ->
         let items = Omake_ast.StringOpExp (".", loc) :: StringIdExp (id, loc) :: items in
            method_id_rev_sequence loc items idl
    | [] ->
         items

let method_id_sequence loc idl =
   List.rev (method_id_rev_sequence loc [] idl)

let method_id_string_exp idl loc =
   Omake_ast.SequenceExp (method_id_sequence loc idl, loc)

let method_id_prefix_string_exp idl loc =
   let idl = List.rev (method_id_rev_sequence loc [StringOpExp (".", loc)] idl) in
   Omake_ast.SequenceExp (idl, loc)

let var_quote (strategy, s, loc) =
   Omake_ast.KeyExp (strategy, s, loc), loc

(*
 * Convert to a body flag and text.
 *)
let get_optcolon_text opt loc =
   match opt with
      None ->
         Omake_ast.OptBody, Omake_ast.NullExp loc
    | Some (body, arg) ->
         body, arg
(*
 * A 3-place rule.
 *)
let rule3 multiple (target, loc1) _ pattern source loc2 body =
   let loc = Lm_location.union_loc loc1 loc2 in
      match pattern with
         Some (pattern, _) ->
            Omake_ast.RuleExp (multiple, target, pattern, source, body, loc)
       | None ->
            RuleExp (multiple, target, NullExp loc2, source, body, loc)

let rule2 multiple target ploc source loc2 body =
   rule3 multiple target ploc None source loc2 body
# 182 "omake_ast_parse.ml"
let yytransl_const = [|
    0|]

let yytransl_block = [|
  257 (* TokEof *);
  258 (* TokEol *);
  259 (* TokWhite *);
  260 (* TokLeftParen *);
  261 (* TokRightParen *);
  262 (* TokArrow *);
  263 (* TokComma *);
  264 (* TokColon *);
  265 (* TokDoubleColon *);
  266 (* TokNamedColon *);
  267 (* TokDollar *);
  268 (* TokEq *);
  269 (* TokArray *);
  270 (* TokDot *);
  271 (* TokId *);
  272 (* TokKey *);
  273 (* TokKeyword *);
  274 (* TokCatch *);
  275 (* TokClass *);
  276 (* TokOp *);
  277 (* TokInt *);
  278 (* TokFloat *);
  279 (* TokString *);
  280 (* TokBeginQuote *);
  281 (* TokEndQuote *);
  282 (* TokBeginQuoteString *);
  283 (* TokEndQuoteString *);
  284 (* TokStringQuote *);
  285 (* TokVar *);
  286 (* TokVarQuote *);
    0|]

let yylhs = "\255\255\
\003\000\003\000\002\000\002\000\005\000\005\000\005\000\005\000\
\005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
\005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
\005\000\005\000\005\000\005\000\005\000\005\000\005\000\001\000\
\021\000\021\000\021\000\022\000\023\000\023\000\023\000\023\000\
\023\000\023\000\015\000\015\000\029\000\028\000\030\000\031\000\
\032\000\024\000\033\000\033\000\026\000\026\000\026\000\026\000\
\011\000\011\000\011\000\016\000\016\000\016\000\012\000\012\000\
\013\000\013\000\036\000\038\000\035\000\035\000\037\000\037\000\
\037\000\037\000\019\000\039\000\039\000\039\000\039\000\039\000\
\039\000\039\000\039\000\017\000\017\000\020\000\020\000\048\000\
\048\000\048\000\004\000\027\000\027\000\018\000\018\000\050\000\
\050\000\007\000\007\000\052\000\052\000\052\000\052\000\052\000\
\014\000\055\000\055\000\009\000\009\000\025\000\025\000\057\000\
\057\000\057\000\059\000\059\000\060\000\058\000\058\000\062\000\
\063\000\063\000\063\000\063\000\063\000\065\000\066\000\061\000\
\061\000\064\000\064\000\064\000\064\000\067\000\067\000\069\000\
\069\000\070\000\070\000\068\000\068\000\072\000\076\000\076\000\
\076\000\054\000\054\000\054\000\034\000\049\000\049\000\049\000\
\049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
\049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
\049\000\049\000\049\000\056\000\056\000\056\000\056\000\056\000\
\056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
\056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
\047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
\047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
\047\000\047\000\047\000\051\000\051\000\051\000\051\000\051\000\
\051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
\051\000\051\000\051\000\051\000\051\000\053\000\053\000\053\000\
\053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
\053\000\053\000\053\000\053\000\053\000\053\000\077\000\077\000\
\077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
\077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
\077\000\077\000\075\000\075\000\075\000\075\000\075\000\075\000\
\075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
\075\000\075\000\075\000\075\000\074\000\074\000\074\000\074\000\
\074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
\071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
\071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
\071\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
\073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
\073\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
\040\000\040\000\040\000\040\000\040\000\040\000\042\000\042\000\
\042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
\042\000\042\000\042\000\042\000\042\000\042\000\041\000\041\000\
\041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
\041\000\041\000\041\000\041\000\044\000\044\000\044\000\044\000\
\044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
\044\000\044\000\043\000\043\000\043\000\043\000\043\000\043\000\
\043\000\043\000\043\000\043\000\043\000\043\000\043\000\046\000\
\046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
\046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
\046\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
\045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
\045\000\045\000\008\000\008\000\010\000\010\000\006\000\006\000\
\000\000\000\000\000\000"

let yylen = "\002\000\
\001\000\003\000\001\000\002\000\002\000\004\000\003\000\007\000\
\011\000\003\000\004\000\004\000\005\000\004\000\005\000\006\000\
\007\000\006\000\008\000\008\000\004\000\006\000\004\000\004\000\
\006\000\006\000\005\000\004\000\009\000\002\000\002\000\002\000\
\000\000\002\000\002\000\004\000\007\000\009\000\001\000\003\000\
\003\000\001\000\001\000\001\000\002\000\001\000\001\000\001\000\
\001\000\001\000\001\000\003\000\001\000\001\000\001\000\001\000\
\000\000\002\000\002\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\002\000\002\000\001\000\002\000\002\000\002\000\
\002\000\002\000\001\000\001\000\002\000\002\000\002\000\002\000\
\002\000\002\000\002\000\001\000\003\000\001\000\003\000\000\000\
\001\000\001\000\001\000\000\000\002\000\001\000\002\000\001\000\
\002\000\000\000\001\000\001\000\001\000\002\000\002\000\002\000\
\001\000\001\000\002\000\001\000\001\000\001\000\002\000\001\000\
\001\000\003\000\001\000\003\000\004\000\001\000\003\000\002\000\
\001\000\001\000\001\000\003\000\004\000\002\000\002\000\001\000\
\002\000\002\000\002\000\001\000\002\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\003\000\000\000\002\000\
\002\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\000\000\002\000\001\000\003\000\000\000\001\000\
\002\000\002\000\002\000"

let yydefred = "\000\000\
\033\000\000\000\000\000\000\000\185\001\000\000\003\000\184\001\
\075\001\076\001\077\001\078\001\000\000\079\001\080\001\081\001\
\069\000\084\001\000\000\000\000\000\000\074\001\082\001\083\001\
\085\001\092\000\092\000\042\000\039\000\046\000\186\001\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\086\001\000\000\
\000\000\062\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\076\000\001\000\187\001\000\000\000\000\032\000\035\000\
\000\000\000\000\034\000\000\000\000\000\148\000\147\000\146\000\
\072\000\000\000\000\000\000\000\074\000\000\000\073\000\000\000\
\000\000\000\000\004\000\005\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\030\000\000\000\031\000\000\000\000\000\
\045\000\144\001\146\001\147\001\148\001\149\001\150\001\151\001\
\152\001\155\001\156\001\157\001\158\001\145\001\153\001\154\001\
\159\001\161\001\049\000\160\001\082\000\163\001\164\001\165\001\
\166\001\167\001\168\001\169\001\172\001\173\001\174\001\175\001\
\162\001\170\001\171\001\176\001\178\001\177\001\081\000\067\000\
\104\001\105\001\106\001\071\000\107\001\110\001\111\001\112\001\
\113\001\103\001\108\001\109\001\114\001\116\001\115\001\077\000\
\087\001\089\001\090\001\091\001\092\001\093\001\096\001\097\001\
\098\001\099\001\088\001\094\001\095\001\100\001\102\001\101\001\
\078\000\068\000\132\001\133\001\134\001\135\001\136\001\137\001\
\053\000\140\001\054\000\055\000\056\000\131\001\138\001\139\001\
\141\001\143\001\070\000\142\001\079\000\117\001\119\001\120\001\
\121\001\122\001\123\001\124\001\127\001\118\001\125\001\126\001\
\128\001\130\001\129\001\080\000\193\000\195\000\196\000\197\000\
\198\000\199\000\200\000\201\000\202\000\205\000\206\000\207\000\
\208\000\194\000\203\000\204\000\209\000\211\000\210\000\083\000\
\000\000\150\000\152\000\153\000\154\000\155\000\156\000\157\000\
\158\000\159\000\160\000\161\000\162\000\165\000\166\000\167\000\
\168\000\151\000\163\000\164\000\169\000\171\000\170\000\093\000\
\213\000\214\000\215\000\216\000\217\000\218\000\219\000\220\000\
\223\000\224\000\225\000\226\000\212\000\221\000\222\000\227\000\
\229\000\228\000\000\000\096\000\000\000\000\000\231\000\232\000\
\233\000\234\000\235\000\236\000\237\000\240\000\241\000\242\000\
\243\000\230\000\238\000\239\000\244\000\000\000\246\000\245\000\
\000\000\100\000\101\000\000\000\007\000\180\001\000\000\010\000\
\149\000\059\000\058\000\040\000\041\000\000\000\000\000\109\000\
\000\000\000\000\115\000\118\000\000\000\000\000\000\000\000\000\
\084\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\002\000\097\000\000\000\000\000\006\000\102\000\103\000\
\104\000\000\000\000\000\143\000\030\001\031\001\032\001\033\001\
\034\001\035\001\000\000\029\001\036\001\037\001\038\001\040\001\
\000\000\039\001\120\000\000\000\000\000\000\000\132\000\139\000\
\138\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\061\000\047\000\089\000\090\000\000\000\086\000\011\000\
\173\000\174\000\175\000\176\000\177\000\178\000\179\000\180\000\
\181\000\182\000\183\000\186\000\187\000\188\000\189\000\172\000\
\184\000\185\000\190\000\000\000\192\000\191\000\000\000\106\000\
\000\000\028\000\000\000\012\000\014\000\000\000\021\000\000\000\
\023\000\000\000\024\000\000\000\036\000\000\000\000\000\000\000\
\000\000\000\000\000\000\127\000\126\000\011\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\022\001\023\001\024\001\
\025\001\012\001\020\001\021\001\026\001\028\001\027\001\133\000\
\141\000\140\000\042\001\043\001\044\001\045\001\046\001\047\001\
\048\001\051\001\052\001\053\001\054\001\041\001\049\001\050\001\
\055\001\057\001\056\001\131\000\134\000\135\000\059\001\060\001\
\061\001\000\000\062\001\063\001\064\001\067\001\068\001\069\001\
\070\001\058\001\065\001\066\001\071\001\073\001\072\001\130\000\
\137\000\136\000\000\000\000\000\000\000\000\000\119\000\000\000\
\116\000\000\000\027\000\000\000\013\000\107\000\000\000\085\000\
\015\000\000\000\000\000\000\000\000\000\110\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\247\000\142\000\249\000\
\250\000\251\000\252\000\253\000\254\000\255\000\000\001\001\001\
\004\001\005\001\006\001\007\001\248\000\002\001\003\001\008\001\
\010\001\009\001\145\000\144\000\000\000\000\000\000\000\018\000\
\000\000\000\000\128\000\000\000\087\000\016\000\000\000\022\000\
\025\000\026\000\037\000\111\000\000\000\052\000\008\000\000\000\
\000\000\000\000\182\001\000\000\129\000\000\000\017\000\000\000\
\000\000\020\000\019\000\000\000\038\000\000\000\029\000\000\000\
\009\000"

let yydgoto = "\004\000\
\005\000\031\000\052\000\053\000\032\000\057\000\022\001\067\000\
\039\001\212\001\072\000\034\000\035\000\124\001\036\000\037\000\
\048\001\049\001\038\000\102\001\006\000\059\000\238\000\142\001\
\231\001\081\001\054\000\107\000\041\000\042\000\043\000\239\000\
\144\001\232\001\045\000\046\000\047\000\048\000\049\000\050\000\
\144\000\161\000\181\000\196\000\127\000\109\000\216\000\103\001\
\240\000\003\001\004\001\025\001\026\001\007\002\127\001\128\001\
\040\001\041\001\042\001\043\001\010\002\044\001\083\001\084\001\
\085\001\086\001\011\002\029\002\208\001\087\001\189\001\169\001\
\210\001\089\001\170\001\147\001\004\002"

let yysindex = "\146\001\
\000\000\153\255\041\255\000\000\000\000\013\001\000\000\000\000\
\000\000\000\000\000\000\000\000\046\255\000\000\000\000\000\000\
\000\000\000\000\105\000\059\255\044\255\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\067\255\
\146\255\129\000\113\000\111\255\144\255\043\000\000\000\159\255\
\000\000\000\000\121\255\177\003\178\002\206\002\203\255\014\000\
\122\002\000\000\000\000\000\000\211\255\010\002\000\000\000\000\
\031\003\197\255\000\000\208\255\203\003\000\000\000\000\000\000\
\000\000\035\000\228\255\046\255\000\000\001\000\000\000\183\255\
\156\000\238\001\000\000\000\000\046\255\046\255\046\255\046\255\
\046\255\046\255\046\255\000\000\046\255\000\000\046\255\046\255\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\040\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\122\002\000\000\046\255\046\255\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\048\000\000\000\000\000\
\038\002\000\000\000\000\046\255\000\000\000\000\046\255\000\000\
\000\000\000\000\000\000\000\000\000\000\054\003\050\000\000\000\
\009\255\062\000\000\000\000\000\081\003\233\255\045\000\151\255\
\000\000\065\000\036\001\194\255\063\000\201\255\075\000\012\000\
\092\000\000\000\000\000\109\000\067\000\000\000\000\000\000\000\
\000\000\111\000\117\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\046\255\000\000\000\000\000\000\000\000\000\000\
\046\255\000\000\000\000\150\002\104\003\127\003\000\000\000\000\
\000\000\046\255\046\255\046\255\046\255\112\000\136\000\044\255\
\179\000\000\000\000\000\000\000\000\000\021\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\189\000\000\000\000\000\010\002\000\000\
\046\255\000\000\046\255\000\000\000\000\208\000\000\000\046\255\
\000\000\046\255\000\000\046\255\000\000\220\000\216\000\235\000\
\046\255\046\255\066\002\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\046\255\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\001\061\255\104\003\054\003\000\000\009\255\
\000\000\046\255\000\000\046\255\000\000\000\000\094\001\000\000\
\000\000\022\000\064\000\085\000\000\000\000\000\254\000\046\255\
\067\000\067\000\015\001\004\001\252\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\104\003\046\255\046\255\000\000\
\046\255\150\002\000\000\012\001\000\000\000\000\018\001\000\000\
\000\000\000\000\000\000\000\000\220\000\000\000\000\000\046\255\
\150\002\019\001\000\000\024\001\000\000\046\255\000\000\022\001\
\026\001\000\000\000\000\030\001\000\000\046\255\000\000\032\001\
\000\000"

let yyrindex = "\000\000\
\000\000\033\001\084\255\000\000\000\000\004\003\000\000\000\000\
\000\000\000\000\000\000\000\000\057\001\000\000\000\000\000\000\
\000\000\000\000\107\255\048\001\046\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\226\255\060\000\097\000\000\000\000\000\000\000\152\001\
\017\255\000\000\000\000\000\000\068\000\094\000\122\000\127\000\
\062\255\000\000\000\000\000\000\000\000\035\001\000\000\000\000\
\210\000\000\000\000\000\000\000\188\255\000\000\000\000\000\000\
\000\000\000\000\000\000\033\001\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\185\000\222\000\123\001\064\001\
\222\000\033\001\123\001\000\000\181\001\000\000\181\001\181\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\245\000\000\000\210\001\239\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\079\001\000\000\000\000\185\000\000\000\000\000\057\001\000\000\
\000\000\000\000\000\000\000\000\000\000\081\001\000\000\000\000\
\082\001\083\001\000\000\000\000\099\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\110\000\000\000\110\000\000\000\
\110\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\233\002\000\000\000\000\000\000\000\000\000\000\
\233\002\000\000\000\000\062\001\065\001\068\001\000\000\000\000\
\000\000\185\255\150\003\150\003\150\003\000\000\000\000\000\000\
\144\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\087\001\000\000\
\123\001\000\000\222\000\000\000\000\000\000\000\000\000\222\000\
\000\000\222\000\000\000\222\000\000\000\085\001\045\255\175\255\
\143\000\048\001\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\233\002\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\077\255\000\000\000\000\000\000\000\000\086\001\
\000\000\185\000\000\000\222\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\094\002\000\000\000\000\150\003\
\000\000\000\000\090\001\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\073\001\033\001\007\000\000\000\
\033\001\005\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\085\001\000\000\000\000\085\001\
\078\001\000\000\000\000\000\000\000\000\143\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\143\000\000\000\000\000\
\000\000"

let yygindex = "\000\000\
\000\000\000\000\000\000\000\000\000\000\254\255\000\000\000\000\
\237\254\125\254\000\000\049\001\050\001\217\254\074\001\076\001\
\176\255\253\255\103\001\000\000\000\000\000\000\008\000\000\000\
\134\000\210\255\104\000\002\000\000\000\000\000\000\000\072\000\
\000\000\190\255\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\147\001\000\000\
\255\000\000\000\136\001\000\000\000\000\239\255\000\000\000\000\
\201\000\091\001\000\000\097\001\088\000\114\001\000\000\000\000\
\000\000\000\000\128\001\095\001\000\000\000\000\000\000\231\254\
\000\000\000\000\000\000\000\000\000\000"

let yytablesize = 1257
let yytable = "\033\000\
\179\000\068\000\058\000\040\000\052\001\035\001\054\001\056\001\
\066\001\039\000\060\000\134\001\088\001\236\001\091\001\092\001\
\066\000\070\000\044\000\048\000\048\000\048\000\048\000\048\000\
\044\000\034\001\044\000\048\000\044\000\048\000\048\000\048\000\
\048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
\048\000\051\000\048\000\027\001\048\000\048\000\048\000\051\000\
\008\000\051\000\106\000\125\000\142\000\159\000\178\000\194\000\
\214\000\071\000\051\000\190\001\209\001\008\000\008\002\075\000\
\001\001\030\001\009\002\075\000\023\001\075\000\075\000\075\000\
\069\000\044\000\038\001\045\001\046\001\047\001\181\001\050\001\
\051\001\053\001\181\001\055\001\057\001\092\000\092\000\092\000\
\092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
\092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
\092\000\092\000\092\000\092\000\179\001\092\000\183\001\092\000\
\092\000\092\000\108\000\126\000\143\000\160\000\180\000\195\000\
\215\000\003\002\083\000\090\000\091\000\092\000\093\000\094\000\
\002\001\073\000\074\000\013\000\024\001\095\000\096\000\097\000\
\098\000\099\000\100\000\101\000\102\000\103\000\104\000\105\000\
\026\000\084\000\027\000\076\000\028\000\029\000\030\000\085\000\
\130\001\007\000\036\002\008\000\009\000\010\000\011\000\012\000\
\131\001\089\000\040\002\013\000\014\000\015\000\016\000\017\000\
\018\000\019\000\020\000\021\000\022\000\023\000\024\000\025\000\
\026\000\050\000\027\000\050\000\028\000\029\000\030\000\015\002\
\032\001\033\001\183\001\190\001\088\001\098\000\183\001\184\001\
\183\001\183\001\183\001\135\001\183\001\169\000\012\002\171\000\
\172\000\173\000\137\001\131\001\005\001\162\000\163\000\164\000\
\165\000\166\000\131\001\006\001\217\000\013\000\063\001\167\000\
\168\000\169\000\170\000\171\000\172\000\173\000\174\000\175\000\
\176\000\177\000\026\000\060\000\027\000\029\001\028\000\029\000\
\030\000\060\000\104\001\190\001\105\001\106\001\107\001\108\001\
\109\001\110\001\111\001\013\000\112\001\113\001\114\001\115\001\
\116\001\117\001\118\001\119\001\120\001\121\001\122\001\123\001\
\026\000\060\001\027\000\061\001\028\000\029\000\030\000\065\001\
\183\001\117\000\214\000\117\000\183\001\139\001\143\001\031\001\
\182\000\183\000\184\000\185\000\186\000\131\001\219\001\016\002\
\013\000\038\001\187\000\188\000\067\001\189\000\220\001\131\001\
\001\001\190\000\191\000\192\000\193\000\026\000\028\001\027\000\
\058\001\028\000\029\000\030\000\086\000\080\001\040\000\057\000\
\057\000\062\001\087\000\088\000\039\000\125\001\090\001\226\001\
\129\001\227\001\125\001\228\001\057\000\061\000\057\000\057\000\
\057\000\017\002\132\001\061\000\093\001\063\000\136\001\063\000\
\148\001\131\001\215\000\063\000\063\000\063\000\149\001\063\000\
\063\000\169\000\138\001\171\000\172\000\173\000\018\002\211\001\
\213\001\214\001\214\001\166\001\186\001\206\001\131\001\064\000\
\002\001\064\000\047\000\140\001\088\000\064\000\064\000\064\000\
\047\000\064\000\064\000\061\000\088\000\082\001\141\001\084\000\
\062\000\063\000\064\000\145\001\044\000\126\001\065\000\084\000\
\146\001\081\000\126\001\065\000\082\000\065\000\223\001\224\001\
\066\000\065\000\065\000\065\000\077\000\065\000\066\000\066\000\
\066\000\078\000\066\000\230\001\079\000\080\000\235\001\237\001\
\183\001\060\000\001\000\002\000\003\000\069\000\183\001\183\001\
\183\001\060\000\001\002\167\001\187\001\207\001\218\000\219\000\
\220\000\221\000\222\000\223\000\224\000\225\000\013\000\226\000\
\227\000\228\000\229\000\230\000\231\000\232\000\233\000\234\000\
\235\000\236\000\237\000\026\000\036\001\027\000\218\001\028\000\
\029\000\030\000\021\002\022\002\183\001\183\001\221\001\005\002\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\225\001\183\001\094\000\183\001\183\001\183\001\038\001\
\013\002\094\000\002\002\094\000\186\001\080\001\229\001\183\001\
\233\001\183\001\183\001\183\001\183\001\214\001\125\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\095\000\183\001\
\234\001\183\001\183\001\183\001\095\000\183\001\095\000\183\001\
\183\001\183\001\019\002\026\002\027\002\023\002\028\002\062\000\
\063\000\064\000\024\002\006\002\186\001\055\000\056\000\008\000\
\030\002\166\001\230\001\031\002\034\002\033\002\062\000\063\000\
\064\000\035\002\037\002\235\001\187\001\082\001\038\002\039\002\
\166\001\041\002\183\001\235\001\091\000\133\001\126\001\105\001\
\106\001\107\001\108\001\109\001\110\001\111\001\013\000\112\001\
\113\001\114\001\115\001\116\001\117\001\118\001\119\001\120\001\
\121\001\122\001\123\001\026\000\183\001\027\000\183\001\028\000\
\029\000\030\000\121\000\121\000\121\000\122\000\122\000\122\000\
\123\000\123\000\123\000\183\001\187\001\124\000\124\000\124\000\
\099\000\167\001\125\000\125\000\125\000\108\000\112\000\113\000\
\105\000\183\001\114\000\181\001\025\002\097\001\098\001\014\002\
\167\001\105\001\106\001\107\001\108\001\109\001\110\001\111\001\
\013\000\112\001\113\001\114\001\115\001\116\001\117\001\118\001\
\119\001\120\001\121\001\122\001\123\001\026\000\099\001\027\000\
\100\001\028\000\029\000\030\000\183\001\222\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\101\001\183\001\059\001\183\001\183\001\
\183\001\043\000\032\002\049\000\049\000\049\000\049\000\043\000\
\064\001\043\000\049\000\043\000\049\000\049\000\049\000\049\000\
\049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
\020\002\049\000\168\001\049\000\049\000\049\000\183\001\216\001\
\183\001\183\001\183\001\183\001\183\001\217\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\215\001\183\001\000\000\
\183\001\183\001\183\001\183\001\188\001\183\001\183\001\183\001\
\183\001\000\000\000\000\000\000\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\000\000\183\001\000\000\183\001\183\001\183\001\
\218\000\219\000\220\000\221\000\222\000\223\000\224\000\225\000\
\013\000\226\000\227\000\228\000\229\000\230\000\231\000\232\000\
\233\000\234\000\235\000\236\000\237\000\026\000\000\000\027\000\
\037\001\028\000\029\000\030\000\218\000\219\000\220\000\221\000\
\222\000\223\000\224\000\225\000\013\000\226\000\227\000\228\000\
\229\000\230\000\231\000\232\000\233\000\234\000\235\000\236\000\
\237\000\026\000\000\000\027\000\000\000\028\000\029\000\030\000\
\033\001\241\000\242\000\243\000\244\000\062\000\063\000\064\000\
\013\000\245\000\246\000\247\000\248\000\249\000\250\000\251\000\
\252\000\253\000\254\000\255\000\000\001\026\000\000\000\027\000\
\000\000\028\000\029\000\030\000\238\001\068\001\239\001\240\001\
\241\001\242\001\243\001\244\001\013\000\245\001\246\001\247\001\
\248\001\249\001\250\001\251\001\252\001\253\001\254\001\255\001\
\000\002\026\000\000\000\027\000\000\000\028\000\029\000\030\000\
\149\000\149\000\184\001\000\000\000\000\149\000\149\000\149\000\
\149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
\149\000\149\000\149\000\149\000\149\000\149\000\000\000\149\000\
\000\000\149\000\149\000\149\000\197\000\198\000\199\000\200\000\
\201\000\000\000\000\000\000\000\013\000\202\000\203\000\204\000\
\205\000\206\000\207\000\208\000\209\000\210\000\211\000\212\000\
\213\000\026\000\000\000\027\000\000\000\028\000\029\000\030\000\
\150\001\068\001\000\000\000\000\000\000\151\001\152\001\153\001\
\013\000\154\001\155\001\156\001\157\001\158\001\159\001\160\001\
\161\001\162\001\163\001\164\001\165\001\026\000\000\000\027\000\
\000\000\028\000\029\000\030\000\128\000\000\000\129\000\130\000\
\131\000\000\000\000\000\000\000\013\000\000\000\000\000\132\000\
\133\000\134\000\135\000\136\000\137\000\138\000\139\000\140\000\
\141\000\026\000\000\000\027\000\000\000\028\000\029\000\030\000\
\145\000\000\000\146\000\147\000\148\000\000\000\000\000\000\000\
\013\000\000\000\000\000\149\000\150\000\151\000\152\000\153\000\
\154\000\155\000\156\000\157\000\158\000\026\000\000\000\027\000\
\000\000\028\000\029\000\030\000\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\000\000\183\001\000\000\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\000\000\000\000\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\000\000\183\001\000\000\183\001\
\183\001\183\001\241\000\242\000\243\000\244\000\000\000\000\000\
\000\000\013\000\245\000\246\000\247\000\248\000\249\000\250\000\
\251\000\252\000\253\000\254\000\255\000\000\001\026\000\000\000\
\027\000\068\001\028\000\029\000\030\000\069\001\070\001\071\001\
\013\000\072\001\073\001\074\001\169\000\075\001\171\000\172\000\
\173\000\076\001\077\001\078\001\079\001\026\000\000\000\027\000\
\000\000\028\000\029\000\030\000\009\000\010\000\011\000\012\000\
\000\000\000\000\000\000\013\000\014\000\015\000\016\000\017\000\
\018\000\094\001\095\001\096\001\022\000\023\000\024\000\025\000\
\026\000\000\000\027\000\068\001\028\000\029\000\030\000\171\001\
\172\001\173\001\013\000\174\001\175\001\176\001\177\001\178\001\
\179\001\180\001\181\001\182\001\183\001\184\001\185\001\026\000\
\000\000\027\000\068\001\028\000\029\000\030\000\191\001\192\001\
\193\001\013\000\194\001\195\001\196\001\197\001\198\001\199\001\
\200\001\201\001\202\001\203\001\204\001\205\001\026\000\000\000\
\027\000\183\001\028\000\029\000\030\000\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\183\001\183\001\
\183\001\183\001\183\001\183\001\183\001\183\001\000\000\183\001\
\000\000\183\001\183\001\183\001\110\000\111\000\112\000\113\000\
\000\000\000\000\000\000\013\000\000\000\114\000\115\000\116\000\
\117\000\118\000\119\000\120\000\121\000\122\000\123\000\124\000\
\026\000\000\000\027\000\000\000\028\000\029\000\030\000\007\001\
\008\001\009\001\062\000\063\000\064\000\013\000\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\026\000\000\000\027\000\000\000\028\000\029\000\
\030\000"

let yycheck = "\002\000\
\047\000\019\000\006\000\002\000\085\000\072\000\087\000\088\000\
\028\001\002\000\013\000\051\001\038\001\145\001\006\001\007\001\
\019\000\020\000\002\001\003\001\004\001\005\001\006\001\007\001\
\008\001\072\000\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\023\001\
\024\001\001\001\026\001\061\000\028\001\029\001\030\001\003\001\
\003\001\005\001\043\000\044\000\045\000\046\000\047\000\048\000\
\049\000\014\001\014\001\085\001\086\001\003\001\002\001\002\001\
\057\000\068\000\006\001\001\001\061\000\008\001\009\001\010\001\
\014\001\002\000\077\000\078\000\079\000\080\000\002\001\082\000\
\083\000\085\000\006\001\087\000\088\000\002\001\003\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\015\001\016\001\017\001\018\001\019\001\020\001\
\021\001\022\001\023\001\024\001\002\001\026\001\004\001\028\001\
\029\001\030\001\043\000\044\000\045\000\046\000\047\000\048\000\
\049\000\147\001\012\001\003\001\004\001\005\001\006\001\007\001\
\057\000\026\000\027\000\011\001\061\000\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\023\001\
\024\001\002\001\026\001\002\001\028\001\029\001\030\001\008\001\
\002\001\001\001\030\002\003\001\004\001\005\001\006\001\007\001\
\010\001\003\001\038\002\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\023\001\
\024\001\003\001\026\001\005\001\028\001\029\001\030\001\223\001\
\002\001\003\001\002\001\213\001\214\001\002\001\006\001\004\001\
\008\001\009\001\010\001\002\001\012\001\015\001\218\001\017\001\
\018\001\019\001\002\001\010\001\008\001\003\001\004\001\005\001\
\006\001\007\001\010\001\004\001\002\001\011\001\025\001\013\001\
\014\001\015\001\016\001\017\001\018\001\019\001\020\001\021\001\
\022\001\023\001\024\001\002\001\026\001\002\001\028\001\029\001\
\030\001\008\001\002\001\005\002\004\001\005\001\006\001\007\001\
\008\001\009\001\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\023\001\
\024\001\005\001\026\001\006\001\028\001\029\001\030\001\025\001\
\002\001\005\001\003\001\007\001\006\001\002\001\061\001\015\001\
\003\001\004\001\005\001\006\001\007\001\010\001\002\001\002\001\
\011\001\028\001\013\001\014\001\031\001\016\001\010\001\010\001\
\025\001\020\001\021\001\022\001\023\001\024\001\004\001\026\001\
\001\001\028\001\029\001\030\001\002\001\038\001\045\001\002\001\
\003\001\002\001\008\001\009\001\045\001\046\001\005\001\136\001\
\012\001\138\001\051\001\140\001\015\001\002\001\017\001\018\001\
\019\001\002\001\002\001\008\001\007\001\002\001\008\001\004\001\
\075\001\010\001\003\001\008\001\009\001\010\001\081\001\012\001\
\013\001\015\001\008\001\017\001\018\001\019\001\002\001\090\001\
\091\001\092\001\093\001\084\001\085\001\086\001\010\001\002\001\
\025\001\004\001\002\001\008\001\002\001\008\001\009\001\010\001\
\008\001\012\001\013\001\003\001\010\001\038\001\002\001\002\001\
\008\001\009\001\010\001\005\001\045\001\046\001\014\001\010\001\
\004\001\009\001\051\001\002\001\012\001\014\001\129\001\131\001\
\002\001\008\001\009\001\010\001\004\001\012\001\008\001\009\001\
\010\001\009\001\012\001\142\001\012\001\013\001\145\001\146\001\
\002\001\002\001\001\000\002\000\003\000\014\001\008\001\009\001\
\010\001\010\001\147\001\084\001\085\001\086\001\003\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\015\001\016\001\017\001\018\001\019\001\020\001\
\021\001\022\001\023\001\024\001\025\001\026\001\004\001\028\001\
\029\001\030\001\233\001\234\001\004\001\005\001\002\001\194\001\
\008\001\009\001\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\023\001\
\024\001\002\001\026\001\002\001\028\001\029\001\030\001\218\001\
\220\001\008\001\147\001\010\001\213\001\214\001\003\001\002\001\
\009\001\004\001\005\001\006\001\007\001\232\001\223\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\002\001\026\001\
\014\001\028\001\029\001\030\001\008\001\015\001\010\001\017\001\
\018\001\019\001\005\001\006\002\007\002\002\001\009\002\008\001\
\009\001\010\001\015\001\012\001\005\002\001\001\002\001\003\001\
\005\001\010\002\021\002\002\001\002\001\024\002\008\001\009\001\
\010\001\002\001\005\001\030\002\213\001\214\001\005\001\002\001\
\025\002\002\001\002\001\038\002\002\001\002\001\223\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\015\001\016\001\017\001\018\001\019\001\020\001\
\021\001\022\001\023\001\024\001\004\001\026\001\015\001\028\001\
\029\001\030\001\005\001\006\001\007\001\005\001\006\001\007\001\
\005\001\006\001\007\001\012\001\005\002\005\001\006\001\007\001\
\002\001\010\002\005\001\006\001\007\001\005\001\005\001\005\001\
\002\001\005\001\005\001\002\001\005\002\045\001\045\001\002\001\
\025\002\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\045\001\026\001\
\045\001\028\001\029\001\030\001\002\001\127\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\015\001\016\001\017\001\018\001\019\001\020\001\021\001\
\022\001\023\001\024\001\045\001\026\001\003\001\028\001\029\001\
\030\001\002\001\021\002\004\001\005\001\006\001\007\001\008\001\
\025\001\010\001\011\001\012\001\013\001\014\001\015\001\016\001\
\017\001\018\001\019\001\020\001\021\001\022\001\023\001\024\001\
\232\001\026\001\084\001\028\001\029\001\030\001\002\001\093\001\
\004\001\005\001\006\001\007\001\008\001\093\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\022\001\023\001\024\001\092\001\026\001\255\255\
\028\001\029\001\030\001\002\001\085\001\004\001\005\001\006\001\
\007\001\255\255\255\255\255\255\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\022\001\
\023\001\024\001\255\255\026\001\255\255\028\001\029\001\030\001\
\003\001\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\255\255\026\001\
\027\001\028\001\029\001\030\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\022\001\
\023\001\024\001\255\255\026\001\255\255\028\001\029\001\030\001\
\003\001\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\255\255\026\001\
\255\255\028\001\029\001\030\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\022\001\
\023\001\024\001\255\255\026\001\255\255\028\001\029\001\030\001\
\003\001\004\001\005\001\255\255\255\255\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\255\255\026\001\
\255\255\028\001\029\001\030\001\003\001\004\001\005\001\006\001\
\007\001\255\255\255\255\255\255\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\022\001\
\023\001\024\001\255\255\026\001\255\255\028\001\029\001\030\001\
\003\001\004\001\255\255\255\255\255\255\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\255\255\026\001\
\255\255\028\001\029\001\030\001\003\001\255\255\005\001\006\001\
\007\001\255\255\255\255\255\255\011\001\255\255\255\255\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\022\001\
\023\001\024\001\255\255\026\001\255\255\028\001\029\001\030\001\
\003\001\255\255\005\001\006\001\007\001\255\255\255\255\255\255\
\011\001\255\255\255\255\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\255\255\026\001\
\255\255\028\001\029\001\030\001\004\001\005\001\006\001\007\001\
\008\001\009\001\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\023\001\
\024\001\255\255\026\001\255\255\028\001\029\001\030\001\004\001\
\005\001\006\001\007\001\008\001\255\255\255\255\011\001\012\001\
\013\001\014\001\015\001\016\001\017\001\018\001\019\001\020\001\
\021\001\022\001\023\001\024\001\255\255\026\001\255\255\028\001\
\029\001\030\001\004\001\005\001\006\001\007\001\255\255\255\255\
\255\255\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\022\001\023\001\024\001\255\255\
\026\001\004\001\028\001\029\001\030\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\255\255\026\001\
\255\255\028\001\029\001\030\001\004\001\005\001\006\001\007\001\
\255\255\255\255\255\255\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\023\001\
\024\001\255\255\026\001\004\001\028\001\029\001\030\001\008\001\
\009\001\010\001\011\001\012\001\013\001\014\001\015\001\016\001\
\017\001\018\001\019\001\020\001\021\001\022\001\023\001\024\001\
\255\255\026\001\004\001\028\001\029\001\030\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\022\001\023\001\024\001\255\255\
\026\001\004\001\028\001\029\001\030\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\023\001\024\001\255\255\026\001\
\255\255\028\001\029\001\030\001\004\001\005\001\006\001\007\001\
\255\255\255\255\255\255\011\001\255\255\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\023\001\
\024\001\255\255\026\001\255\255\028\001\029\001\030\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\015\001\016\001\017\001\018\001\019\001\020\001\021\001\
\022\001\023\001\024\001\255\255\026\001\255\255\028\001\029\001\
\030\001"

let yynames_const = "\
  "

let yynames_block = "\
  TokEof\000\
  TokEol\000\
  TokWhite\000\
  TokLeftParen\000\
  TokRightParen\000\
  TokArrow\000\
  TokComma\000\
  TokColon\000\
  TokDoubleColon\000\
  TokNamedColon\000\
  TokDollar\000\
  TokEq\000\
  TokArray\000\
  TokDot\000\
  TokId\000\
  TokKey\000\
  TokKeyword\000\
  TokCatch\000\
  TokClass\000\
  TokOp\000\
  TokInt\000\
  TokFloat\000\
  TokString\000\
  TokBeginQuote\000\
  TokEndQuote\000\
  TokBeginQuoteString\000\
  TokEndQuoteString\000\
  TokStringQuote\000\
  TokVar\000\
  TokVarQuote\000\
  "

let yyact = [|
  (fun _ -> failwith "parser")
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 214 "omake_ast_parse.mly"
          ( raise End_of_file )
# 939 "omake_ast_parse.ml"
               : Omake_ast.body_flag * Omake_ast.exp))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'text) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 216 "omake_ast_parse.mly"
          ( NoBody, sequence_exp _1 _2 )
# 948 "omake_ast_parse.ml"
               : Omake_ast.body_flag * Omake_ast.exp))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 224 "omake_ast_parse.mly"
          ( raise End_of_file )
# 955 "omake_ast_parse.ml"
               : Omake_ast.body_flag * Omake_ast.exp))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'shell_line) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 226 "omake_ast_parse.mly"
          ( _1 )
# 963 "omake_ast_parse.ml"
               : Omake_ast.body_flag * Omake_ast.exp))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 232 "omake_ast_parse.mly"
          ( NoBody, sequence_exp [] _2 )
# 971 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'keyword_text_optcolon) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 236 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let body, arg = get_optcolon_text _3 _4 in
            let loc = Lm_location.union_loc loc1 _4 in
               body, CommandExp (Lm_symbol.add id, arg, [], loc)
          )
# 985 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'opt_literal_colon) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 242 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let body = _2 in
            let loc = Lm_location.union_loc loc1 _3 in
            let arg = Omake_ast.NullExp loc in
               body, CommandExp (Lm_symbol.add id, arg, [], loc)
          )
# 999 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 6 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 5 : 'opt_white) in
    let _3 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _4 = (Parsing.peek_val __caml_parser_env 3 : 'opt_args) in
    let _5 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _6 = (Parsing.peek_val __caml_parser_env 1 : 'opt_colon) in
    let _7 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 251 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let body = _6 in
            let loc = Lm_location.union_loc loc1 _7 in
	    let args = args_of_parse_args _4 in
            let e = Omake_ast.ApplyExp (CommandApply, Lm_symbol.add id, args, loc) in
               body, e
          )
# 1018 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 10 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 9 : 'opt_white) in
    let _3 = (Parsing.peek_val __caml_parser_env 8 : string * Lm_location.t) in
    let _4 = (Parsing.peek_val __caml_parser_env 7 : 'opt_white) in
    let _5 = (Parsing.peek_val __caml_parser_env 6 : string * Lm_location.t) in
    let _6 = (Parsing.peek_val __caml_parser_env 5 : 'opt_white) in
    let _7 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _8 = (Parsing.peek_val __caml_parser_env 3 : 'opt_white) in
    let _9 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _10 = (Parsing.peek_val __caml_parser_env 1 : 'opt_colon) in
    let _11 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 261 "omake_ast_parse.mly"
          ( let _, loc1 = _1 in
            let loc = Lm_location.union_loc loc1 _11 in
            let name, _ = _3 in
            let v, _ = _7 in
               _10, CatchExp (Lm_symbol.add name, Lm_symbol.add v, [], loc)
          )
# 1040 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'opt_id_list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 270 "omake_ast_parse.mly"
          ( let _, loc1 = _1 in
            let loc = Lm_location.union_loc loc1 _3 in
               NoBody, ClassExp (List.map Lm_symbol.add _2, loc)
          )
# 1052 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 277 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let loc2 = _4 in
            let id = method_id_intern id in
            let loc = Lm_location.union_loc loc1 loc2 in
            let add_flag = define_flag _2 in
               ColonBody, VarDefBodyExp (id, DefineString, add_flag, [], loc)
          )
# 1068 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'method_id_prefix_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 287 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let loc2 = _4 in
            let id = method_id_intern id in
            let loc = Lm_location.union_loc loc1 loc2 in
            let add_flag = define_flag _2 in
               ColonBody, ObjectDefExp (id, add_flag, [], loc)
          )
# 1084 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'text_nonempty) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 297 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let loc2 = _5 in
            let e = simplify _4 in
            let id = method_id_intern id in
            let loc = Lm_location.union_loc loc1 loc2 in
            let add_flag = define_flag _2 in
               NoBody, VarDefExp (id, DefineString, add_flag, e, loc)
          )
# 1102 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'var_quote_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 308 "omake_ast_parse.mly"
          ( let _, id, loc1 = _1 in
            let loc2 = _4 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let add_flag = define_flag _2 in
               ColonBody, KeyDefBodyExp (id, DefineString, add_flag, [], loc)
          )
# 1117 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : 'var_quote_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'text_nonempty) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 317 "omake_ast_parse.mly"
          ( let _, id, loc1 = _1 in
            let loc2 = _5 in
            let e = simplify _4 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let add_flag = define_flag _2 in
               NoBody, KeyDefExp (id, DefineString, add_flag, e, loc)
          )
# 1134 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 327 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let loc2 = _6 in
            let id = method_id_intern id in
            let loc = Lm_location.union_loc loc1 loc2 in
            let add_flag = define_flag _4 in
               ArrayBody, VarDefBodyExp (id, DefineArray, add_flag, [], loc)
          )
# 1152 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 6 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 5 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 4 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 3 : string * Lm_location.t) in
    let _5 = (Parsing.peek_val __caml_parser_env 2 : 'opt_white) in
    let _6 = (Parsing.peek_val __caml_parser_env 1 : 'text_nonempty) in
    let _7 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 337 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let loc2 = _7 in
            let id = method_id_intern id in
            let loc = Lm_location.union_loc loc1 loc2 in
            let add_flag = define_flag _4 in
               NoBody, VarDefExp (id, DefineArray, add_flag, _6, loc)
          )
# 1171 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'opt_args) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'opt_colon) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 347 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let body = _5 in
            let loc = Lm_location.union_loc loc1 _6 in
            let args = args_of_parse_args _3 in
            let e =
               match id with
                  [id] ->
                     Omake_ast.ApplyExp (CommandApply, Lm_symbol.add id, args, loc)
                | _ ->
                     MethodApplyExp (CommandApply, method_id_intern id, args, loc)
            in
               body, e
          )
# 1195 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 7 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 6 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 5 : 'opt_args) in
    let _4 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _5 = (Parsing.peek_val __caml_parser_env 3 : 'opt_colon) in
    let _6 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _7 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _8 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 363 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let body = _5 in
            let loc = Lm_location.union_loc loc1 _8 in
            let params = get_fun_params _3 in
            let arg = Omake_ast.ArrowArg (params, StringOpExp ("...", loc)) in
            let e =
               match id with
                  [id] ->
                     Omake_ast.ApplyExp (CommandApply, Lm_symbol.add id, [arg], loc)
                | _ ->
                     MethodApplyExp (CommandApply, method_id_intern id, [arg], loc)
            in
               body, e
          )
# 1222 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 7 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 6 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 5 : 'opt_args) in
    let _4 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _5 = (Parsing.peek_val __caml_parser_env 3 : 'opt_white) in
    let _6 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _7 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _8 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 380 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let id = method_id_intern id in
            let params = get_fun_params _3 in
            let loc = Lm_location.union_loc loc1 _8 in
               ColonBody, FunDefExp (id, params, [], loc)
          )
# 1241 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'other_id_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'source) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 389 "omake_ast_parse.mly"
          ( ColonBody, rule2 false _1 _2 _3 _4 [] )
# 1251 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : 'other_id_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'target) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'source) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 391 "omake_ast_parse.mly"
          ( ColonBody, rule3 false _1 _2 _3 _5 _6 [] )
# 1263 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'other_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'source) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 393 "omake_ast_parse.mly"
          ( ColonBody, rule2 false _1 _2 _3 _4 [] )
# 1273 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'other_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'source) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 395 "omake_ast_parse.mly"
          ( ColonBody, rule2 true _1 _2 _3 _4 [] )
# 1283 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : 'other_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'target) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'source) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 397 "omake_ast_parse.mly"
          ( ColonBody, rule3 false _1 _2 _3 _5 _6 [] )
# 1295 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : 'other_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'target) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'source) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 399 "omake_ast_parse.mly"
          ( ColonBody, rule3 true _1 _2 _3 _5 _6 [] )
# 1307 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'source_nonapply) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 407 "omake_ast_parse.mly"
          ( let idl, loc = _1 in
            let e = method_id_string_exp idl loc in
               ColonBody, rule2 true (e, loc) _2 _4 _5 []
          )
# 1321 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'method_id_prefix_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'source) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 412 "omake_ast_parse.mly"
          ( let idl, loc = _1 in
            let e = method_id_prefix_string_exp idl loc in
               ColonBody, rule2 true (e, loc) _2 _3 _4 []
          )
# 1334 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 8 : 'method_id_opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 7 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 6 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 5 : 'method_id_opt_white) in
    let _5 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _6 = (Parsing.peek_val __caml_parser_env 3 : 'opt_args) in
    let _7 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _8 = (Parsing.peek_val __caml_parser_env 1 : 'opt_colon) in
    let _9 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 417 "omake_ast_parse.mly"
          ( let super, loc1 = _1 in
            let name, _ = _4 in
            let body = _8 in
            let loc = Lm_location.union_loc loc1 _9 in
            let args = args_of_parse_args _6 in
            let e =
               match super, name with
                  [super], [name] ->
                     Omake_ast.SuperApplyExp (CommandApply, Lm_symbol.add super, Lm_symbol.add name, args, loc)
                | _, [_] ->
                     raise (Omake_value_type.OmakeException (loc_exp_pos loc, StringStringError ("illegal super class", method_id_string super)))
                | _ ->
                     raise (Omake_value_type.OmakeException (loc_exp_pos loc, StringStringError ("illegal field name", method_id_string name)))
            in
               body, e
          )
# 1364 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'other_id_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 436 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               NoBody, ShellExp (e, loc)
          )
# 1374 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'other_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 440 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               NoBody, ShellExp (e, loc)
          )
# 1384 "omake_ast_parse.ml"
               : 'shell_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_deps) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 450 "omake_ast_parse.mly"
          ( List.rev _1 )
# 1392 "omake_ast_parse.ml"
               : (Omake_ast.exp * Omake_ast.exp * Lm_location.t) list))
; (fun __caml_parser_env ->
    Obj.repr(
# 455 "omake_ast_parse.mly"
          ( [] )
# 1398 "omake_ast_parse.ml"
               : 'rev_deps))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_deps) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'dep) in
    Obj.repr(
# 457 "omake_ast_parse.mly"
          ( _2 :: _1 )
# 1406 "omake_ast_parse.ml"
               : 'rev_deps))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_deps) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 459 "omake_ast_parse.mly"
          ( _1 )
# 1414 "omake_ast_parse.ml"
               : 'rev_deps))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'target) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'target) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Lm_location.t) in
    Obj.repr(
# 464 "omake_ast_parse.mly"
          ( let _, loc2 = _2 in
            let target, loc1 =
               match _1 with
                  Some (e, loc1) -> e, loc1
                | None ->
                     NullExp loc2, loc2
            in
            let source =
               match _3 with
                  Some (e, _) -> e
                | None -> NullExp loc2
            in
            let loc = Lm_location.union_loc loc1 _4 in
               target, source, loc
          )
# 1438 "omake_ast_parse.ml"
               : 'dep))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 6 : string * Omake_ast.apply_strategy * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 5 : 'opt_white) in
    let _3 = (Parsing.peek_val __caml_parser_env 4 : string * Lm_location.t) in
    let _4 = (Parsing.peek_val __caml_parser_env 3 : 'opt_white) in
    let _5 = (Parsing.peek_val __caml_parser_env 2 : 'method_name) in
    let _6 = (Parsing.peek_val __caml_parser_env 1 : 'opt_apply_args) in
    let _7 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 485 "omake_ast_parse.mly"
          ( let _, strategy, loc1 = _1 in
            let _, loc2 = _7 in
            let idl, _ = _5 in
            let args = args_of_parse_args _6 in
            let loc = Lm_location.union_loc loc1 loc2 in
               match idl with
                  [id] ->
                     Omake_ast.ApplyExp (strategy, Lm_symbol.add id, args, loc), loc
                | _ ->
                     MethodApplyExp (strategy, method_id_intern idl, args, loc), loc
          )
# 1461 "omake_ast_parse.ml"
               : 'apply))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 8 : string * Omake_ast.apply_strategy * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 7 : 'opt_white) in
    let _3 = (Parsing.peek_val __caml_parser_env 6 : string * Lm_location.t) in
    let _4 = (Parsing.peek_val __caml_parser_env 5 : 'opt_white) in
    let _5 = (Parsing.peek_val __caml_parser_env 4 : 'id) in
    let _6 = (Parsing.peek_val __caml_parser_env 3 : string * Lm_location.t) in
    let _7 = (Parsing.peek_val __caml_parser_env 2 : 'id) in
    let _8 = (Parsing.peek_val __caml_parser_env 1 : 'opt_apply_args) in
    let _9 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 497 "omake_ast_parse.mly"
          ( let _, strategy, loc1 = _1 in
            let _, loc2 = _9 in
            let super, _ = _5 in
            let v, _ = _7 in
            let args = args_of_parse_args _8 in
            let loc = Lm_location.union_loc loc1 loc2 in
               SuperApplyExp (strategy, Lm_symbol.add super, Lm_symbol.add v, args, loc), loc
          )
# 1483 "omake_ast_parse.ml"
               : 'apply))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Omake_ast.apply_strategy * string * Lm_location.t) in
    Obj.repr(
# 506 "omake_ast_parse.mly"
          ( let strategy, id, loc = _1 in
               ApplyExp (strategy, Lm_symbol.add id, [], loc), loc
          )
# 1492 "omake_ast_parse.ml"
               : 'apply))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'rev_text) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 510 "omake_ast_parse.mly"
          ( let id1, loc1 = _1 in
            let id2, loc2 = _3 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let el = Omake_ast.StringOtherExp (id1, loc1) :: List.rev_append _2 
            [ Omake_ast.StringOtherExp (id2, loc2)] in
               QuoteExp (el, loc), loc
          )
# 1507 "omake_ast_parse.ml"
               : 'apply))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'rev_text) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 518 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let _, loc2 = _3 in
            let loc = Lm_location.union_loc loc1 loc2 in
               QuoteStringExp (id.[0], List.rev _2, loc), loc
          )
# 1520 "omake_ast_parse.ml"
               : 'apply))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 524 "omake_ast_parse.mly"
          ( let s, loc = _1 in
               QuoteExp ([StringOtherExp (s, loc)], loc), loc
          )
# 1529 "omake_ast_parse.ml"
               : 'apply))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'var_quote) in
    Obj.repr(
# 534 "omake_ast_parse.mly"
          ( _1 )
# 1536 "omake_ast_parse.ml"
               : 'var_quote_opt_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'var_quote_white) in
    Obj.repr(
# 536 "omake_ast_parse.mly"
          ( let strategy, id, _, loc = _1 in
               strategy, id, loc
          )
# 1545 "omake_ast_parse.ml"
               : 'var_quote_opt_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'var_quote) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 543 "omake_ast_parse.mly"
          ( let strategy, id, loc = _1 in
            let s, _ = _2 in
               strategy, id, s, loc
          )
# 1556 "omake_ast_parse.ml"
               : 'var_quote_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Omake_ast.apply_strategy * string * Lm_location.t) in
    Obj.repr(
# 551 "omake_ast_parse.mly"
          ( _1 )
# 1563 "omake_ast_parse.ml"
               : 'var_quote))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'var_quote_opt_white) in
    Obj.repr(
# 559 "omake_ast_parse.mly"
          ( var_quote _1 )
# 1570 "omake_ast_parse.ml"
               : 'quote_opt_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'var_quote_white) in
    Obj.repr(
# 564 "omake_ast_parse.mly"
          ( let strategy, id, s, loc = _1 in
            let e, _ = var_quote (strategy, id, loc) in
               e, s, loc
          )
# 1580 "omake_ast_parse.ml"
               : 'quote_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'var_quote) in
    Obj.repr(
# 571 "omake_ast_parse.mly"
          ( var_quote _1 )
# 1587 "omake_ast_parse.ml"
               : 'quote))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_method_name) in
    Obj.repr(
# 579 "omake_ast_parse.mly"
          ( let idl, loc = _1 in
               List.rev idl, loc
          )
# 1596 "omake_ast_parse.ml"
               : 'method_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'id) in
    Obj.repr(
# 586 "omake_ast_parse.mly"
          ( let id, loc = _1 in
               [id], loc
          )
# 1605 "omake_ast_parse.ml"
               : 'rev_method_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'rev_method_name) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'id) in
    Obj.repr(
# 590 "omake_ast_parse.mly"
          ( let idl, loc1 = _1 in
            let id, loc2 = _3 in
               id :: idl, Lm_location.union_loc loc1 loc2
          )
# 1617 "omake_ast_parse.ml"
               : 'rev_method_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 597 "omake_ast_parse.mly"
          ( _1 )
# 1624 "omake_ast_parse.ml"
               : 'id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 599 "omake_ast_parse.mly"
          ( _1 )
# 1631 "omake_ast_parse.ml"
               : 'id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 601 "omake_ast_parse.mly"
          ( _1 )
# 1638 "omake_ast_parse.ml"
               : 'id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 603 "omake_ast_parse.mly"
          ( _1 )
# 1645 "omake_ast_parse.ml"
               : 'id))
; (fun __caml_parser_env ->
    Obj.repr(
# 608 "omake_ast_parse.mly"
          ( [] )
# 1651 "omake_ast_parse.ml"
               : 'opt_id_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'opt_id_list) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'white) in
    Obj.repr(
# 610 "omake_ast_parse.mly"
          ( _1 )
# 1659 "omake_ast_parse.ml"
               : 'opt_id_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'opt_id_list) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'id) in
    Obj.repr(
# 612 "omake_ast_parse.mly"
          ( let id, _ = _2 in
               id :: _1
          )
# 1669 "omake_ast_parse.ml"
               : 'opt_id_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'method_id_opt_white) in
    Obj.repr(
# 624 "omake_ast_parse.mly"
          ( let idl, loc = _1 in
               method_id_string_exp idl loc, loc
          )
# 1678 "omake_ast_parse.ml"
               : 'other_id_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'method_id_prefix_opt_white) in
    Obj.repr(
# 628 "omake_ast_parse.mly"
          ( let idl, loc = _1 in
               method_id_prefix_string_exp idl loc, loc
          )
# 1687 "omake_ast_parse.ml"
               : 'other_id_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote_opt_white) in
    Obj.repr(
# 632 "omake_ast_parse.mly"
          ( _1 )
# 1694 "omake_ast_parse.ml"
               : 'other_id_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_method_id) in
    Obj.repr(
# 637 "omake_ast_parse.mly"
          ( let id, loc = _1 in
               List.rev id, loc
          )
# 1703 "omake_ast_parse.ml"
               : 'method_id_opt_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_method_id_white) in
    Obj.repr(
# 641 "omake_ast_parse.mly"
          ( let id, _, loc = _1 in
               List.rev id, loc
          )
# 1712 "omake_ast_parse.ml"
               : 'method_id_opt_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_method_id_prefix) in
    Obj.repr(
# 648 "omake_ast_parse.mly"
          ( let id, loc = _1 in
               List.rev id, loc
          )
# 1721 "omake_ast_parse.ml"
               : 'method_id_prefix_opt_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_method_id_prefix_white) in
    Obj.repr(
# 652 "omake_ast_parse.mly"
          ( let id, _, loc = _1 in
               List.rev id, loc
          )
# 1730 "omake_ast_parse.ml"
               : 'method_id_prefix_opt_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_method_id) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 659 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let s, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               id, s, loc
          )
# 1742 "omake_ast_parse.ml"
               : 'rev_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_method_id_prefix) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 668 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let s, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               id, s, loc
          )
# 1754 "omake_ast_parse.ml"
               : 'rev_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 677 "omake_ast_parse.mly"
          ( let id, loc = _1 in
               [id], loc
          )
# 1763 "omake_ast_parse.ml"
               : 'rev_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_method_id_prefix) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'id) in
    Obj.repr(
# 681 "omake_ast_parse.mly"
          ( let idl, loc1 = _1 in
            let id, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               id :: idl, loc
          )
# 1775 "omake_ast_parse.ml"
               : 'rev_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_method_id) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 690 "omake_ast_parse.mly"
          ( let idl, loc1 = _1 in
            let _, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               idl, loc
          )
# 1787 "omake_ast_parse.ml"
               : 'rev_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 696 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let _, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               [id], loc
          )
# 1799 "omake_ast_parse.ml"
               : 'rev_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 702 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let _, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               [id], loc
          )
# 1811 "omake_ast_parse.ml"
               : 'rev_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 708 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let _, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               [id], loc
          )
# 1823 "omake_ast_parse.ml"
               : 'rev_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_other_target) in
    Obj.repr(
# 737 "omake_ast_parse.mly"
          ( let l, loc = _1 in
               sequence_exp (List.rev l) loc, loc
          )
# 1832 "omake_ast_parse.ml"
               : 'other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'other_start) in
    Obj.repr(
# 744 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               [e], loc
          )
# 1841 "omake_ast_parse.ml"
               : 'rev_other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_method_id) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'other_method_id) in
    Obj.repr(
# 748 "omake_ast_parse.mly"
          ( let idl, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let el = [e; method_id_string_exp (List.rev idl) loc1] in
               el, loc
          )
# 1854 "omake_ast_parse.ml"
               : 'rev_other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_method_id_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'other_method_id_white) in
    Obj.repr(
# 755 "omake_ast_parse.mly"
          ( let idl, s, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let el = [e; Omake_ast.StringWhiteExp (s, loc1); method_id_string_exp (List.rev idl) loc1] in
               el, loc
          )
# 1867 "omake_ast_parse.ml"
               : 'rev_other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_method_id_prefix) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'other_method_id_prefix) in
    Obj.repr(
# 762 "omake_ast_parse.mly"
          ( let idl, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let el = [e; method_id_prefix_string_exp (List.rev idl) loc1] in
               el, loc
          )
# 1880 "omake_ast_parse.ml"
               : 'rev_other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_method_id_prefix_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'other_method_id_prefix_white) in
    Obj.repr(
# 769 "omake_ast_parse.mly"
          ( let idl, s, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let el = [e; Omake_ast.StringWhiteExp (s, loc1); method_id_prefix_string_exp (List.rev idl) loc1] in
               el, loc
          )
# 1893 "omake_ast_parse.ml"
               : 'rev_other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'quote) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'other_quote_id) in
    Obj.repr(
# 776 "omake_ast_parse.mly"
          ( let id, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let el = [id; e] in
               el, loc
          )
# 1906 "omake_ast_parse.ml"
               : 'rev_other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'quote_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'other_quote_id_white) in
    Obj.repr(
# 783 "omake_ast_parse.mly"
          ( let id, s, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
            let el = [id; StringWhiteExp (s, loc1); e] in
               el, loc
          )
# 1919 "omake_ast_parse.ml"
               : 'rev_other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_other_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'target_next) in
    Obj.repr(
# 790 "omake_ast_parse.mly"
          ( let el, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               e :: el, loc
          )
# 1931 "omake_ast_parse.ml"
               : 'rev_other_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'target) in
    Obj.repr(
# 802 "omake_ast_parse.mly"
         ( match _1 with
              Some (e, _) ->
                 Lm_symbol.SymbolTable.add Lm_symbol.SymbolTable.empty Omake_symbol.normal_sym e
            | None ->
                 Lm_symbol.SymbolTable.empty
         )
# 1943 "omake_ast_parse.ml"
               : 'source))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'source) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'target) in
    Obj.repr(
# 809 "omake_ast_parse.mly"
         ( let table = _1 in
           let name, _ = _2 in
              match _3 with
                 Some (e, _) ->
                    Lm_symbol.SymbolTable.add table (Lm_symbol.add name) e
               | None ->
                    table
         )
# 1959 "omake_ast_parse.ml"
               : 'source))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'source_target) in
    Obj.repr(
# 824 "omake_ast_parse.mly"
         ( match _1 with
              Some (e, _) ->
                 Lm_symbol.SymbolTable.add Lm_symbol.SymbolTable.empty Omake_symbol.normal_sym e
            | None ->
                 Lm_symbol.SymbolTable.empty
         )
# 1971 "omake_ast_parse.ml"
               : 'source_nonapply))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'source_nonapply) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'target) in
    Obj.repr(
# 831 "omake_ast_parse.mly"
         ( let table = _1 in
           let name, _ = _2 in
              match _3 with
                 Some (e, _) ->
                    Lm_symbol.SymbolTable.add table (Lm_symbol.add name) e
               | None ->
                    table
         )
# 1987 "omake_ast_parse.ml"
               : 'source_nonapply))
; (fun __caml_parser_env ->
    Obj.repr(
# 843 "omake_ast_parse.mly"
          ( None )
# 1993 "omake_ast_parse.ml"
               : 'source_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'other_id_target) in
    Obj.repr(
# 845 "omake_ast_parse.mly"
          ( Some _1 )
# 2000 "omake_ast_parse.ml"
               : 'source_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'other_target) in
    Obj.repr(
# 847 "omake_ast_parse.mly"
          ( Some _1 )
# 2007 "omake_ast_parse.ml"
               : 'source_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_text) in
    Obj.repr(
# 861 "omake_ast_parse.mly"
          ( List.rev _1 )
# 2014 "omake_ast_parse.ml"
               : 'text))
; (fun __caml_parser_env ->
    Obj.repr(
# 866 "omake_ast_parse.mly"
          ( [] )
# 2020 "omake_ast_parse.ml"
               : 'rev_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_text) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_next) in
    Obj.repr(
# 868 "omake_ast_parse.mly"
          ( let e, _ = _2 in
               e :: _1
          )
# 2030 "omake_ast_parse.ml"
               : 'rev_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 881 "omake_ast_parse.mly"
          ( None )
# 2037 "omake_ast_parse.ml"
               : 'target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'rev_target) in
    Obj.repr(
# 883 "omake_ast_parse.mly"
          ( let l, loc = _2 in
               Some (sequence_exp (List.rev l) loc, loc)
          )
# 2047 "omake_ast_parse.ml"
               : 'target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'target_start) in
    Obj.repr(
# 890 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               [e], loc
          )
# 2056 "omake_ast_parse.ml"
               : 'rev_target))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_target) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'target_next) in
    Obj.repr(
# 894 "omake_ast_parse.mly"
          ( let l, loc1 = _1 in
            let e, loc2 = _2 in
               e :: l, Lm_location.union_loc loc1 loc2
          )
# 2067 "omake_ast_parse.ml"
               : 'rev_target))
; (fun __caml_parser_env ->
    Obj.repr(
# 907 "omake_ast_parse.mly"
          ( None )
# 2073 "omake_ast_parse.ml"
               : 'keyword_text_optcolon))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_keyword_text) in
    Obj.repr(
# 909 "omake_ast_parse.mly"
          ( let code, _, el, loc = _1 in
               Some (code, sequence_exp (List.rev el) loc)
          )
# 2082 "omake_ast_parse.ml"
               : 'keyword_text_optcolon))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'keyword_target_start) in
    Obj.repr(
# 916 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               OptBody, [], [e], loc
          )
# 2091 "omake_ast_parse.ml"
               : 'rev_keyword_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'colon) in
    Obj.repr(
# 920 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               ColonBody, [e], [], loc
          )
# 2100 "omake_ast_parse.ml"
               : 'rev_keyword_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_keyword_text) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'white) in
    Obj.repr(
# 924 "omake_ast_parse.mly"
          ( let code, final, prefix, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               code, e :: final, prefix, loc
          )
# 2112 "omake_ast_parse.ml"
               : 'rev_keyword_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_keyword_text) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'target_start) in
    Obj.repr(
# 930 "omake_ast_parse.mly"
          ( let _, final, prefix, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               OptBody, [], e :: (final @ prefix), loc
          )
# 2124 "omake_ast_parse.ml"
               : 'rev_keyword_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_keyword_text) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'colon) in
    Obj.repr(
# 936 "omake_ast_parse.mly"
          ( let _, final, prefix, loc1 = _1 in
            let e, loc2 = _2 in
            let loc = Lm_location.union_loc loc1 loc2 in
               ColonBody, [e], final @ prefix, loc
          )
# 2136 "omake_ast_parse.ml"
               : 'rev_keyword_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_text_nonempty) in
    Obj.repr(
# 948 "omake_ast_parse.mly"
          ( let l, loc = _1 in
               sequence_exp (List.rev l) loc
          )
# 2145 "omake_ast_parse.ml"
               : 'text_nonempty))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'text_nonwhite) in
    Obj.repr(
# 955 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               [e], loc
          )
# 2154 "omake_ast_parse.ml"
               : 'rev_text_nonempty))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_text_nonempty) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_next) in
    Obj.repr(
# 959 "omake_ast_parse.mly"
          ( let l, loc1 = _1 in
            let e, loc2 = _2 in
               e :: l, Lm_location.union_loc loc1 loc2
          )
# 2165 "omake_ast_parse.ml"
               : 'rev_text_nonempty))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 970 "omake_ast_parse.mly"
          ( [] )
# 2172 "omake_ast_parse.ml"
               : 'opt_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'args) in
    Obj.repr(
# 972 "omake_ast_parse.mly"
          ( _1 )
# 2179 "omake_ast_parse.ml"
               : 'opt_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 977 "omake_ast_parse.mly"
          ( [] )
# 2186 "omake_ast_parse.ml"
               : 'opt_apply_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'white) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'args) in
    Obj.repr(
# 979 "omake_ast_parse.mly"
          ( _2 )
# 2194 "omake_ast_parse.ml"
               : 'opt_apply_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_args) in
    Obj.repr(
# 983 "omake_ast_parse.mly"
          ( List.rev _1 )
# 2201 "omake_ast_parse.ml"
               : 'args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_arrow_args) in
    Obj.repr(
# 985 "omake_ast_parse.mly"
          ( List.rev _1 )
# 2208 "omake_ast_parse.ml"
               : 'args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'rev_arrow_args) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'rev_args) in
    Obj.repr(
# 987 "omake_ast_parse.mly"
          ( List.rev_append _1 (List.rev _3) )
# 2217 "omake_ast_parse.ml"
               : 'args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arrow_arg) in
    Obj.repr(
# 992 "omake_ast_parse.mly"
          ( [_1] )
# 2224 "omake_ast_parse.ml"
               : 'rev_arrow_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'rev_arrow_args) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'arrow_arg) in
    Obj.repr(
# 994 "omake_ast_parse.mly"
          ( _3 :: _1 )
# 2233 "omake_ast_parse.ml"
               : 'rev_arrow_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'rev_args) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'rev_any_arg) in
    Obj.repr(
# 999 "omake_ast_parse.mly"
          ( let el, loc2 = _4 in
               NormalArg (ArrowArg (get_fun_params (List.rev _1), sequence_exp (List.rev el) loc2))
          )
# 2245 "omake_ast_parse.ml"
               : 'arrow_arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arg) in
    Obj.repr(
# 1006 "omake_ast_parse.mly"
          ( [_1] )
# 2252 "omake_ast_parse.ml"
               : 'rev_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'rev_args) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'arg) in
    Obj.repr(
# 1008 "omake_ast_parse.mly"
          ( _3 :: _1 )
# 2261 "omake_ast_parse.ml"
               : 'rev_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'arg_inner) in
    Obj.repr(
# 1012 "omake_ast_parse.mly"
          ( _2 )
# 2269 "omake_ast_parse.ml"
               : 'arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'rev_normal_arg) in
    Obj.repr(
# 1017 "omake_ast_parse.mly"
          ( let el, loc = _1 in
            let e = sequence_exp (List.rev el) loc in
               NormalArg (ExpArg e)
          )
# 2279 "omake_ast_parse.ml"
               : 'arg_inner))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arg_id) in
    Obj.repr(
# 1022 "omake_ast_parse.mly"
          ( let (id, _), w, loc = _1 in
	       IdArg (id, w, loc)
          )
# 2288 "omake_ast_parse.ml"
               : 'arg_inner))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arg_key) in
    Obj.repr(
# 1026 "omake_ast_parse.mly"
          ( let (id, _), w, loc = _1 in
	       IdArg (id, w, loc)
          )
# 2297 "omake_ast_parse.ml"
               : 'arg_inner))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'arg_key) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 1030 "omake_ast_parse.mly"
          ( let (id, _), _, loc1 = _1 in
	    let key = key_of_id id in
               NormalArg (KeyArg (key, NullExp loc1))
          )
# 2309 "omake_ast_parse.ml"
               : 'arg_inner))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'arg_key) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'opt_white) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'rev_any_arg) in
    Obj.repr(
# 1035 "omake_ast_parse.mly"
          ( let (id, _), _, _ = _1 in
	    let key = key_of_id id in
            let el, loc2 = _4 in
               NormalArg (KeyArg (key, sequence_exp (List.rev el) loc2))
          )
# 2323 "omake_ast_parse.ml"
               : 'arg_inner))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'id) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 1043 "omake_ast_parse.mly"
          ( let id = _1 in
            let _, loc = id in
               id, _2, loc
          )
# 2334 "omake_ast_parse.ml"
               : 'arg_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 1051 "omake_ast_parse.mly"
          ( let id = _1 in
            let _, loc = id in
               id, _2, loc
          )
# 2345 "omake_ast_parse.ml"
               : 'arg_key))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg_any_start) in
    Obj.repr(
# 1059 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               [e], loc
          )
# 2354 "omake_ast_parse.ml"
               : 'rev_any_arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_any_arg) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg_next) in
    Obj.repr(
# 1063 "omake_ast_parse.mly"
          ( let l, loc1 = _1 in
            let e, loc2 = _2 in
               e :: l, Lm_location.union_loc loc1 loc2
          )
# 2365 "omake_ast_parse.ml"
               : 'rev_any_arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'arg_key) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg_next_noneq) in
    Obj.repr(
# 1071 "omake_ast_parse.mly"
          ( let (id, loc0), w, loc1 = _1 in
            let id = Omake_ast.StringIdExp (id, loc0) in
            let e, loc2 = _2 in
            let el =
               match w with
                  Some (w, loc0) ->
                     [e; Omake_ast.StringWhiteExp (w, loc0); id]
                | None ->
                     [e; id]
            in
               el, Lm_location.union_loc loc1 loc2
          )
# 2384 "omake_ast_parse.ml"
               : 'rev_normal_arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'arg_id) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg_any_start) in
    Obj.repr(
# 1084 "omake_ast_parse.mly"
          ( let (id, loc0), w, loc1 = _1 in
            let id = Omake_ast.StringIdExp (id, loc0) in
            let e, loc2 = _2 in
            let el =
               match w with
                  Some (w, loc3) ->
                     [e; StringWhiteExp (w, loc3); id]
                | None ->
                     [e; id]
            in
               el, Lm_location.union_loc loc1 loc2
          )
# 2403 "omake_ast_parse.ml"
               : 'rev_normal_arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg_start) in
    Obj.repr(
# 1097 "omake_ast_parse.mly"
          ( let e, loc = _1 in
               [e], loc
          )
# 2412 "omake_ast_parse.ml"
               : 'rev_normal_arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_normal_arg) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg_next) in
    Obj.repr(
# 1101 "omake_ast_parse.mly"
          ( let l, loc1 = _1 in
            let e, loc2 = _2 in
               e :: l, Lm_location.union_loc loc1 loc2
          )
# 2423 "omake_ast_parse.ml"
               : 'rev_normal_arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arg_any_start) in
    Obj.repr(
# 1109 "omake_ast_parse.mly"
   ( _1 )
# 2430 "omake_ast_parse.ml"
               : 'paren_arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg) in
    Obj.repr(
# 1111 "omake_ast_parse.mly"
   ( _1 )
# 2437 "omake_ast_parse.ml"
               : 'paren_arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arg_next_noneq) in
    Obj.repr(
# 1116 "omake_ast_parse.mly"
   ( _1 )
# 2444 "omake_ast_parse.ml"
               : 'paren_arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg) in
    Obj.repr(
# 1118 "omake_ast_parse.mly"
   ( _1 )
# 2451 "omake_ast_parse.ml"
               : 'paren_arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arg_start) in
    Obj.repr(
# 1123 "omake_ast_parse.mly"
   ( _1 )
# 2458 "omake_ast_parse.ml"
               : 'paren_arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg) in
    Obj.repr(
# 1125 "omake_ast_parse.mly"
   ( _1 )
# 2465 "omake_ast_parse.ml"
               : 'paren_arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arg_next) in
    Obj.repr(
# 1130 "omake_ast_parse.mly"
   ( _1 )
# 2472 "omake_ast_parse.ml"
               : 'paren_arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg) in
    Obj.repr(
# 1132 "omake_ast_parse.mly"
   ( _1 )
# 2479 "omake_ast_parse.ml"
               : 'paren_arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * Lm_location.t) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'rev_paren_text) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1137 "omake_ast_parse.mly"
          ( let s1, loc1 = _1 in
            let sl = _2 in
            let s3, loc3 = _3 in
            let loc = Lm_location.union_loc loc1 loc3 in
            let el = Omake_ast.StringOpExp (s1, loc1) :: (List.rev (Omake_ast.StringOpExp (s3, loc3) :: sl)) in
               SequenceExp (el, loc), loc
          )
# 2494 "omake_ast_parse.ml"
               : 'paren_arg))
; (fun __caml_parser_env ->
    Obj.repr(
# 1148 "omake_ast_parse.mly"
   ( [] )
# 2500 "omake_ast_parse.ml"
               : 'rev_paren_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_paren_text) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_next) in
    Obj.repr(
# 1150 "omake_ast_parse.mly"
   ( let s, _ = _2 in
               s :: _1
          )
# 2510 "omake_ast_parse.ml"
               : 'rev_paren_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'rev_paren_text) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_arg) in
    Obj.repr(
# 1154 "omake_ast_parse.mly"
          ( let s, _ = _2 in
               s :: _1
          )
# 2520 "omake_ast_parse.ml"
               : 'rev_paren_text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1164 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2527 "omake_ast_parse.ml"
               : 'colon))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1166 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2534 "omake_ast_parse.ml"
               : 'colon))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1168 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2541 "omake_ast_parse.ml"
               : 'colon))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1172 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringWhiteExp (s, loc), loc )
# 2548 "omake_ast_parse.ml"
               : 'white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1176 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringWhiteExp (s, loc), loc )
# 2555 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1178 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2562 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1180 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2569 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1182 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2576 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1184 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2583 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1186 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2590 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1188 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2597 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1190 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2604 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1192 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2611 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1194 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2618 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1196 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2625 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1198 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2632 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1200 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 2639 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1202 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 2646 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1204 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 2653 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1206 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2660 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1208 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2667 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1210 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2674 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1212 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2681 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1214 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 2688 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1216 "omake_ast_parse.mly"
  ( _1 )
# 2695 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1218 "omake_ast_parse.mly"
  ( _1 )
# 2702 "omake_ast_parse.ml"
               : 'text_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1222 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2709 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1224 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2716 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1226 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2723 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1228 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2730 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1230 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2737 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1232 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2744 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1234 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2751 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1236 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2758 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1238 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2765 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1240 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2772 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1242 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2779 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1244 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 2786 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1246 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 2793 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1248 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 2800 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1250 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2807 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1252 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2814 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1254 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2821 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1256 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2828 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1258 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 2835 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1260 "omake_ast_parse.mly"
  ( _1 )
# 2842 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1262 "omake_ast_parse.mly"
  ( _1 )
# 2849 "omake_ast_parse.ml"
               : 'text_nonwhite))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1266 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringWhiteExp (s, loc), loc )
# 2856 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1268 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2863 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1270 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2870 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1272 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2877 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1274 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2884 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1276 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2891 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1278 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2898 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1280 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2905 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1282 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2912 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1284 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 2919 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1286 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 2926 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1288 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 2933 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1290 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2940 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1292 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2947 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1294 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2954 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1296 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 2961 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1298 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 2968 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1300 "omake_ast_parse.mly"
  ( _1 )
# 2975 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1302 "omake_ast_parse.mly"
  ( _1 )
# 2982 "omake_ast_parse.ml"
               : 'target_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1306 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2989 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1308 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 2996 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1310 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3003 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1312 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3010 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1314 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3017 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1316 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3024 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1318 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3031 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1320 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3038 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1322 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 3045 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1324 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3052 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1326 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3059 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1328 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3066 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1330 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3073 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1332 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3080 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1334 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3087 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1336 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3094 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1338 "omake_ast_parse.mly"
  ( _1 )
# 3101 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1340 "omake_ast_parse.mly"
  ( _1 )
# 3108 "omake_ast_parse.ml"
               : 'target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1344 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3115 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1346 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3122 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1348 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3129 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1350 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3136 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1352 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3143 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1354 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3150 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1356 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3157 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1358 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 3164 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1360 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3171 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1362 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3178 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1364 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3185 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1366 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3192 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1368 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3199 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1370 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3206 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1372 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3213 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1374 "omake_ast_parse.mly"
  ( _1 )
# 3220 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1376 "omake_ast_parse.mly"
  ( _1 )
# 3227 "omake_ast_parse.ml"
               : 'keyword_target_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1380 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringWhiteExp (s, loc), loc )
# 3234 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1382 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3241 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1384 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3248 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1386 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3255 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1388 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3262 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1390 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3269 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1392 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3276 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1394 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3283 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1396 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3290 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1398 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3297 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1400 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 3304 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1402 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3311 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1404 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3318 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1406 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3325 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1408 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3332 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1410 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3339 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1412 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3346 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1414 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3353 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1416 "omake_ast_parse.mly"
  ( _1 )
# 3360 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1418 "omake_ast_parse.mly"
  ( _1 )
# 3367 "omake_ast_parse.ml"
               : 'paren_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1422 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringWhiteExp (s, loc), loc )
# 3374 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1424 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3381 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1426 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3388 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1428 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3395 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1430 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3402 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1432 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3409 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1434 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3416 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1436 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3423 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1438 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 3430 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1440 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3437 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1442 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3444 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1444 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3451 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1446 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3458 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1448 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3465 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1450 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3472 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1452 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3479 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1454 "omake_ast_parse.mly"
  ( _1 )
# 3486 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1456 "omake_ast_parse.mly"
  ( _1 )
# 3493 "omake_ast_parse.ml"
               : 'arg_next))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1460 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3500 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1462 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3507 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1464 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3514 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1466 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3521 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1468 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3528 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1470 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3535 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1472 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3542 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1474 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3549 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1476 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3556 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1478 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3563 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1480 "omake_ast_parse.mly"
  ( _1 )
# 3570 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1482 "omake_ast_parse.mly"
  ( _1 )
# 3577 "omake_ast_parse.ml"
               : 'arg_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1486 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3584 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1488 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3591 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1490 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3598 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1492 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3605 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1494 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3612 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1496 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3619 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1498 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3626 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1500 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 3633 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1502 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3640 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1504 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3647 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1506 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3654 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1508 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3661 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1510 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3668 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1512 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3675 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1514 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3682 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1516 "omake_ast_parse.mly"
  ( _1 )
# 3689 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1518 "omake_ast_parse.mly"
  ( _1 )
# 3696 "omake_ast_parse.ml"
               : 'arg_any_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1522 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3703 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1524 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3710 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1526 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3717 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1528 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3724 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1530 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3731 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1532 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3738 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1534 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 3745 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1536 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3752 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1538 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3759 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1540 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3766 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1542 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3773 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1544 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3780 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1546 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3787 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1548 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3794 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1550 "omake_ast_parse.mly"
  ( _1 )
# 3801 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1552 "omake_ast_parse.mly"
  ( _1 )
# 3808 "omake_ast_parse.ml"
               : 'arg_next_noneq))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1556 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3815 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1558 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3822 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1560 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3829 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1562 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3836 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1564 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3843 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1566 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3850 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1568 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3857 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1570 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3864 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1572 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3871 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1574 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3878 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1576 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3885 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1578 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3892 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1580 "omake_ast_parse.mly"
  ( _1 )
# 3899 "omake_ast_parse.ml"
               : 'other_start))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1584 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringWhiteExp (s, loc), loc )
# 3906 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1586 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3913 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1588 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3920 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1590 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3927 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1592 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3934 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1594 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 3941 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1596 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 3948 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1598 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 3955 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1600 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 3962 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1602 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3969 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1604 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3976 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1606 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3983 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1608 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 3990 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1610 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 3997 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1612 "omake_ast_parse.mly"
  ( _1 )
# 4004 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1614 "omake_ast_parse.mly"
  ( _1 )
# 4011 "omake_ast_parse.ml"
               : 'other_method_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1618 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4018 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1620 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4025 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1622 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4032 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1624 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4039 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1626 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 4046 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1628 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 4053 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1630 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 4060 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1632 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4067 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1634 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4074 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1636 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4081 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1638 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4088 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1640 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 4095 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1642 "omake_ast_parse.mly"
  ( _1 )
# 4102 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1644 "omake_ast_parse.mly"
  ( _1 )
# 4109 "omake_ast_parse.ml"
               : 'other_method_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1648 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringWhiteExp (s, loc), loc )
# 4116 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1650 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4123 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1652 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4130 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1654 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4137 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1656 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4144 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1658 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4151 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1660 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4158 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1662 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4165 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1664 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 4172 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1666 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 4179 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1668 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4186 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1670 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 4193 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1672 "omake_ast_parse.mly"
  ( _1 )
# 4200 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1674 "omake_ast_parse.mly"
  ( _1 )
# 4207 "omake_ast_parse.ml"
               : 'other_method_id_prefix_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1678 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4214 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1680 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4221 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1682 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4228 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1684 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4235 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1686 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4242 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1688 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4249 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1690 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4256 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1692 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 4263 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1694 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 4270 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1696 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4277 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1698 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 4284 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1700 "omake_ast_parse.mly"
  ( _1 )
# 4291 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1702 "omake_ast_parse.mly"
  ( _1 )
# 4298 "omake_ast_parse.ml"
               : 'other_method_id_prefix))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1706 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringWhiteExp (s, loc), loc )
# 4305 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1708 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4312 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1710 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4319 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1712 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4326 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1714 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4333 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1716 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4340 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1718 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4347 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1720 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4354 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1722 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 4361 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1724 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 4368 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1726 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 4375 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1728 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4382 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1730 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4389 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1732 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4396 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1734 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4403 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1736 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 4410 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1738 "omake_ast_parse.mly"
  ( _1 )
# 4417 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1740 "omake_ast_parse.mly"
  ( _1 )
# 4424 "omake_ast_parse.ml"
               : 'other_quote_id_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1744 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4431 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1746 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4438 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1748 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4445 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1750 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4452 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1752 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4459 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1754 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4466 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1756 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOpExp (s, loc), loc )
# 4473 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1758 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIdExp (s, loc), loc )
# 4480 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1760 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringIntExp (s, loc), loc )
# 4487 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1762 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringFloatExp (s, loc), loc )
# 4494 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1764 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4501 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1766 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4508 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1768 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4515 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1770 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringKeywordExp (s, loc), loc )
# 4522 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1772 "omake_ast_parse.mly"
  ( let (s, loc) = _1 in StringOtherExp (s, loc), loc )
# 4529 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'quote) in
    Obj.repr(
# 1774 "omake_ast_parse.mly"
  ( _1 )
# 4536 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'apply) in
    Obj.repr(
# 1776 "omake_ast_parse.mly"
  ( _1 )
# 4543 "omake_ast_parse.ml"
               : 'other_quote_id))
; (fun __caml_parser_env ->
    Obj.repr(
# 1784 "omake_ast_parse.mly"
          ( OptBody )
# 4549 "omake_ast_parse.ml"
               : 'opt_literal_colon))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'colon) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 1786 "omake_ast_parse.mly"
          ( ColonBody )
# 4557 "omake_ast_parse.ml"
               : 'opt_literal_colon))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 1791 "omake_ast_parse.mly"
          ( OptBody )
# 4564 "omake_ast_parse.ml"
               : 'opt_colon))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'opt_white) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'colon) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'opt_white) in
    Obj.repr(
# 1793 "omake_ast_parse.mly"
          ( ColonBody )
# 4573 "omake_ast_parse.ml"
               : 'opt_colon))
; (fun __caml_parser_env ->
    Obj.repr(
# 1798 "omake_ast_parse.mly"
          ( None )
# 4579 "omake_ast_parse.ml"
               : 'opt_white))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Lm_location.t) in
    Obj.repr(
# 1800 "omake_ast_parse.mly"
          ( Some _1 )
# 4586 "omake_ast_parse.ml"
               : 'opt_white))
(* Entry deps *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
(* Entry shell *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
(* Entry string *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
|]
let yytables =
  { Parsing.actions=yyact;
    Parsing.transl_const=yytransl_const;
    Parsing.transl_block=yytransl_block;
    Parsing.lhs=yylhs;
    Parsing.len=yylen;
    Parsing.defred=yydefred;
    Parsing.dgoto=yydgoto;
    Parsing.sindex=yysindex;
    Parsing.rindex=yyrindex;
    Parsing.gindex=yygindex;
    Parsing.tablesize=yytablesize;
    Parsing.table=yytable;
    Parsing.check=yycheck;
    Parsing.error_function=parse_error;
    Parsing.names_const=yynames_const;
    Parsing.names_block=yynames_block }
let deps (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 1 lexfun lexbuf : (Omake_ast.exp * Omake_ast.exp * Lm_location.t) list)
let shell (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 2 lexfun lexbuf : Omake_ast.body_flag * Omake_ast.exp)
let string (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 3 lexfun lexbuf : Omake_ast.body_flag * Omake_ast.exp)
