#!/bin/sh

set -e
export LC_ALL=C  # Stable sorting regardless of the locale

. build/autogen.d/geoip

no_geoip=0
cmake=0
while [ $# -gt 0 ]; do
    opt=$1; shift
    if [ "$opt" = "--no-geoip" -o "$opt" = "-n" ]; then
        no_geoip=1
    elif [ "$opt" = "--cmake" ]; then
        cmake=1
    else
        echo "usage: $0 [--cmake] [-n|--no-geoip]" 1>&2
        exit 1
    fi
done

slug() {
    echo $(echo $1|tr '/-' '_'|sed 's/^include_measurement_kit/mk/g')
}

gen_automatic_includes() {

    hh=include/measurement_kit/common/git_version.hpp
    echo "// File autogenerated by \`$0\`, do not edit"                    > $hh
    echo "#ifndef MEASUREMENT_KIT_COMMON_GIT_VERSION_HPP"                 >> $hh
    echo "#define MEASUREMENT_KIT_COMMON_GIT_VERSION_HPP"                 >> $hh
    echo ""                                                               >> $hh
    echo "#define MK_VERSION_FULL \"`git describe --tags`\""              >> $hh
    echo ""                                                               >> $hh
    echo "#endif"                                                         >> $hh

    rm -f include/measurement_kit/*.hpp
    for name in $(ls include/measurement_kit/|grep -v portable); do
        hh=include/measurement_kit/$name.hpp
        echo "// File autogenerated by \`$0\`, do not edit"                > $hh
        echo "#ifndef MEASUREMENT_KIT_$(echo $name|tr 'a-z' 'A-Z')_HPP"   >> $hh
        echo "#define MEASUREMENT_KIT_$(echo $name|tr 'a-z' 'A-Z')_HPP"   >> $hh
        for nn in $(ls include/measurement_kit/$name/); do
            echo "#include <measurement_kit/$name/$nn>"                   >> $hh
        done
        echo "#endif"                                                     >> $hh
    done
}

gen_headers() {
    echo "$(slug $1)_includedir = \${prefix}/$1"
    echo "$(slug $1)_include_HEADERS = # Empty"
    for name in `ls $1`; do
        if [ ! -d $1/$name ]; then
            echo "$(slug $1)_include_HEADERS += $1/$name"
        fi
    done
    echo ""
    for name in `ls $1`; do
        if [ -d $1/$name ]; then
            gen_headers $1/$name
        fi
    done
}

gen_sources() {
    for name in `ls $2`; do
        if [ ! -d $2/$name ]; then
            if echo $name | grep -q '\.c[p]*$'; then
                echo "$1 += $2/$name"
            fi
        fi
    done
    for name in `ls $2`; do
        if [ -d $2/$name ]; then
            gen_sources $1 $2/$name
        fi
    done
}

gen_executables() {
    for name in `ls $2`; do
        if [ ! -d $2/$name ]; then
            if echo $name | grep -q '\.c[p]*$'; then
                bin_name=$(echo $name | sed 's/\.c[p]*$//g')
                echo ""
                echo "if $3"
                echo "    $1 += $2/$bin_name"
                echo "endif"
                echo "/$2/$bin_name" >> .gitignore
                echo "$(slug $2/$bin_name)_SOURCES = $2/$name"
                echo "$(slug $2/$bin_name)_LDADD = libmeasurement_kit.la"
            fi
        fi
    done
    for name in `ls $2`; do
        if [ -d $2/$name ]; then
            gen_executables $1 $2/$name $3
        fi
    done
}

grep -v -E "^/(test|example){1}/.*" .gitignore > .gitignore.new
echo /GeoIP.dat >> .gitignore.new
echo /GeoIPASNum.dat >> .gitignore.new
echo /GeoLiteCity.dat >> .gitignore.new
mv .gitignore.new .gitignore

echo "* Generating automatic includes: include/measurement_kit/*.hpp"
gen_automatic_includes include/measurement_kit

echo "* Generating include.am"
echo "# Autogenerated by $0 on date $(date)"                      > include.am
echo ""                                                          >> include.am
gen_sources libmeasurement_kit_la_SOURCES src/libmeasurement_kit >> include.am
gen_sources measurement_kit_SOURCES src/measurement_kit          >> include.am
echo ""                                                          >> include.am
gen_headers include/measurement_kit                              >> include.am
gen_executables noinst_PROGRAMS example BUILD_EXAMPLES           >> include.am
gen_executables ALL_TESTS test BUILD_TESTS                       >> include.am

echo "* Updating .gitignore"
sort -u .gitignore > .gitignore.new
mv .gitignore.new .gitignore

if [ $no_geoip -ne 1 ]; then
    autogen_get_geoip
fi

if [ $cmake -eq 0 ]; then
    echo "* Running 'autoreconf -i'"
    autoreconf -i
fi

echo ""
echo "=== autogen.sh complete ==="
echo ""
echo "MeasurementKit is now ready to be compiled. To proceed you shall run"
echo "now the './configure' script that adapts the build to your system."
echo ""
echo "The './configure' script shall also check for external dependencies. "
echo "MeasurementKit external dependencies are:"
echo ""
echo "    - GeoIP"
echo "    - openssl / libressl"
echo "    - libevent"
echo ""
echo "If any of these dependencies is missing, the './configure' script shall"
echo "stop and tell you how you could install it."
