package tests::UIIntegerWidgetTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::Config::TypeSpec;
use Lire::UI::Widget;
use Lire::Test::CursesUIDriver;

use Curses::UI;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'driver'} = new Lire::Test::CursesUIDriver();
    $self->{'driver'}->setup_curses_ui();

    $self->{'ui'} = new Curses::UI();
    $self->{'driver'}->set_curses_ui( $self->{'ui'} );
    $self->{'window'} = $self->{'ui'}->add( 'window', 'Window' );

    my $spec = new Lire::Config::ConfigSpec( 'name' => 'lire' );
    $spec->add( new Lire::Config::IntegerSpec( 'name' => 'test',
                                               'section' => 'programs',
                                               'summary' => 'Test Summary',
                                               'description' => '<para>Test Description</para>' ) );
    $self->{'int_value'} = $spec->get( 'test' )->instance( 'value' => '12345' );
    $self->{'spec'} = $spec;

    return;
}


sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    $self->{'driver'}->teardown_curses_ui();

    return;
}

sub test_new {
    my $self = $_[0];

    my $win = $self->{'window'};
    $self->assert_dies( qr{missing 'value' parameter},
                         sub { my $widget = $win->add( 'test_widget',
                                                       'Lire::UI::IntegerWidget' ) } );
    $self->assert_dies( qr{'value' parameter should be a 'Lire::Config::Scalar' instance, not 'HASH},
                      sub { my $widget = $win->add( 'test_widget',
                                                    'Lire::UI::IntegerWidget',
                                                    'Value' => {} ) } );

    my $widget = $win->add( 'test_widget', 'Lire::UI::IntegerWidget',
                            'value' => $self->{'int_value'} );
    $self->assert_isa( 'Lire::UI::IntegerWidget', $widget );

    my $text_entry = $widget->getobj( 'text_entry' );
    $self->assert_isa( 'Curses::UI::TextEntry', $text_entry );
    $self->assert_str_equals( \&Lire::UI::IntegerWidget::_on_change_cb,
                              $text_entry->{'-onchange'} );
    $self->assert_str_equals( '12345', $text_entry->{'-text'} );

    $self->assert_isa( 'Curses::UI::Buttonbox', $widget->getobj( 'buttons' ) );

    return;
}

sub test_on_change_cb {
    my $self = $_[0];

    my $called = 'onvaluechanged not called';
    my $win = $self->{'window'};
    my $widget = $win->add( 'test_widget', 'Lire::UI::IntegerWidget',
                            'value' => $self->{'int_value'},
                            'onvaluechanged' => sub { $called = 'called' } );
    my $text_entry = $widget->getobj( 'text_entry' );
    my $or_value = $text_entry->text();
    $text_entry->text( '654321' );
    $text_entry->run_event( '-onchange' );
    $self->assert_str_not_equals( $or_value, $self->{'int_value'}->get() );
    $self->assert_str_equals( '654321', $self->{'int_value'}->get() );

    $self->{'driver'}->enter_text( '/window/test_widget/text_entry',
                                  "\ca\ck654321abcd" );
    $self->assert_str_equals( '654321', $self->{'int_value'}->get() );
    $self->assert_str_equals( 'called', $called );
}

sub test_refresh_view {
    my $self = $_[0];

    my $widget = $self->{'window'}->add( 'test_widget',
                                         'Lire::UI::IntegerWidget',
                                         'value' => $self->{'int_value'} );

    $self->{'int_value'}->set( '162534' );
    $widget->refresh_view();

    my $text_entry = $widget->getobj( 'text_entry' );
    $self->assert_str_equals( '162534', $text_entry->text() );
}

sub test_decrease {
    my $self = $_[0];

    $self->{'int_value'}->set( '162534' );
    my $widget = $self->{'window'}->add( 'test_widget',
                                         'Lire::UI::IntegerWidget',
                                         'value' => $self->{'int_value'} );
    $widget->getobj( 'buttons' )->{'-buttons'}[0]->{'-onpress'}->();

    my $text_entry = $widget->getobj( 'text_entry' );
    $self->assert_str_equals( '162533', $text_entry->text() );

    $widget->do_routine( 'decrease', 10 );
    $self->assert_str_equals( '162523', $text_entry->text() );
}

sub test_increase {
    my $self = $_[0];

    $self->{'int_value'}->set( '162534' );
    my $widget = $self->{'window'}->add( 'test_widget',
                                         'Lire::UI::IntegerWidget',
                                         'value' => $self->{'int_value'} );

    $widget->getobj( 'buttons' )->{'-buttons'}[1]->{'-onpress'}->();

    my $text_entry = $widget->getobj( 'text_entry' );
    $self->assert_str_equals( '162535', $text_entry->text() );

    $widget->do_routine( 'increase', 10 );
    $self->assert_str_equals( '162545', $text_entry->text() );
}

1;
