# Copyright 2011 OpenStack Foundation
# Copyright 2011 Nebula, Inc.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from keystoneclient import base


class Domain(base.Resource):
    """Represents an Identity domain.

    Attributes:
        * id: a uuid that identifies the domain
        * name: the name of the domain
        * description: a description of the domain
        * enabled: determines whether the domain is enabled

    """

    pass


class DomainManager(base.CrudManager):
    """Manager class for manipulating Identity domains."""

    resource_class = Domain
    collection_key = 'domains'
    key = 'domain'

    def create(self, name, description=None, enabled=True, **kwargs):
        """Create a domain.

        :param str name: the name of the domain.
        :param str description: a description of the domain.
        :param bool enabled: whether the domain is enabled.
        :param kwargs: any other attribute provided will be passed to the
                       server.

        :returns: the created domain returned from server.
        :rtype: :class:`keystoneclient.v3.domains.Domain`

        """
        return super(DomainManager, self).create(
            name=name,
            description=description,
            enabled=enabled,
            **kwargs)

    def get(self, domain):
        """Retrieve a domain.

        :param domain: the domain to be retrieved from the server.
        :type domain: str or :class:`keystoneclient.v3.domains.Domain`

        :returns: the specified domain returned from server.
        :rtype: :class:`keystoneclient.v3.domains.Domain`

        """
        return super(DomainManager, self).get(
            domain_id=base.getid(domain))

    def list(self, **kwargs):
        """List domains.

        :param kwargs: allows filter criteria to be passed where
         supported by the server.

        :returns: a list of domains.
        :rtype: list of :class:`keystoneclient.v3.domains.Domain`.

        """
        # Ref bug #1267530 we have to pass 0 for False to get the expected
        # results on all keystone versions
        if kwargs.get('enabled') is False:
            kwargs['enabled'] = 0
        return super(DomainManager, self).list(**kwargs)

    def update(self, domain, name=None,
               description=None, enabled=None, **kwargs):
        """Update a domain.

        :param domain: the domain to be updated on the server.
        :type domain: str or :class:`keystoneclient.v3.domains.Domain`
        :param str name: the new name of the domain.
        :param str description: the new description of the domain.
        :param bool enabled: whether the domain is enabled.
        :param kwargs: any other attribute provided will be passed to the
                       server.

        :returns: the updated domain returned from server.
        :rtype: :class:`keystoneclient.v3.domains.Domain`

        """
        return super(DomainManager, self).update(
            domain_id=base.getid(domain),
            name=name,
            description=description,
            enabled=enabled,
            **kwargs)

    def delete(self, domain):
        """Delete a domain.

        :param domain: the domain to be deleted on the server.
        :type domain: str or :class:`keystoneclient.v3.domains.Domain`

        :returns: Response object with 204 status.
        :rtype: :class:`requests.models.Response`

        """
        return super(DomainManager, self).delete(
            domain_id=base.getid(domain))
